#-----------------------------------------------------------------------------------------
#
# Veritas System Recovery 18 Automation
#
# Copyright (c) 2018 by Veritas Technologies LLC
# All Rights Reserved
#
# THIS SCRIPTING EXAMPLE IS PROVIDED IN CONJUNCTION WITH YOUR LICENSE FOR VERITAS 
# SYSTEM RECOVERY SOFTWARE PRODUCTS, AND MAY BE USED ONLY IN CONJUNCTION WITH 
# THAT SOFTWARE, UNDER THE TERMS OF THE END USER LICENSE AGREEMENT THAT ACCOMPANIED 
# THE SOFTWARE. THIS SCRIPTING EXAMPLE IS PROPRIETARY VERITAS PROPERTY.  YOU MAY NOT 
# COPY, DISTRIBUTE, LICENSE OR SUBLICENSE THE SCRIPTING DOCUMENTATION.
#
# 
#
#-----------------------------------------------------------------------------------------
#
#   Example Script: Verify a recovery point with chain.
#
#   Objects Used: ProtectorAuto, RestoreImageJob, FolderLocation
#   Initialize the following variables with appropriate values:                               
#               1. sFilePath – set the file path
#-----------------------------------------------------------------------------------------

	$v2iAuto = $null
	$oNet = $null
	$oLocation = $null
	$oVerifyJob = $null
	$sFilePath = $null
	$sUsrName = $null
    $sPwd = $null
	$sImagePwd = $null
    $bNetworkPath = $false
	#----------------------------------------------------------------------------------------------------
    # Function : CreateLocation
    # Input parameters :
    #                   a)  Path can be a local folder location or a network location.
    #                   b)  Domain user name if the path is a network location.
    #                   c)  password for authenticating to the network location.
    #
    # Return parameter: Location COM object containing all the information about the location
    #-----------------------------------------------------------------------------------------------------
    
    function CreateLocation 
    {
        param ([string]$Path, [string]$domainUser, [string]$password)
        
        $oLocation = $null
        $sepPos = $null 
        $dotPos = $null
        $Path = $Path -replace "/","\"
        
        # Parse the location.
        # Check if the location is a network location or a folder location.
        # If it is a network location, set the user name and password.
        if ($Path.SubString(0,2) -eq "\\")
        {
            $oLocation = New-Object -ComObject "Veritas.VProRecovery.NetworkLocation"
            $oLocation.NetworkUser = $domainUser
            $oLocation.NetworkPassword = $password
        }
        else 
        {
            $oLocation = New-Object -ComObject "Veritas.VProRecovery.FolderLocation"
        }
        
        $sepPos = $Path.LastIndexOf("\")
        $dotPos = $Path.LastIndexOf(".")

        $oLocation.Path = [string]$Path.SubString(0, $sepPos)
        $oLocation.FileSpec = [string]$Path.SubString($sepPos+1, $dotPos - $sepPos - 1)
        $oLocation.Extension = [string]$Path.SubString($dotPos)
        
        Write-Host "Path: " $oLocation.Path 
        Write-Host "FileSpecs: " $oLocation.FileSpec 
        Write-Host "Entension: "$oLocation.Extension
        
        return $oLocation
    }
	
    #-----------------------------------------------------------------------------------------
	# Step 1: Process command line arguments.
	#-----------------------------------------------------------------------------------------
	
	if ([int]$args.Length -lt 1) 
	{
		Write-Host "Usage: VerifyImage.ps1 [LOCAL/NETWORK file Path]"
		exit $1;
	}
    
    $sFilePath = $args[0]

    #-----------------------------------------------------------------------------------------
	# Validating input file for v2i/iv2i extension.
	#-----------------------------------------------------------------------------------------
    $dotIdx = $sFilePath.LastIndexOf(".")
    if ($dotIdx -lt 0)
    {
        Write-Host "Wrong Input"
        Write-Host "Usage: VerifyImage.ps1 [LOCAL/NETWORK file Path]. For example: E:\Backup\machine_name_xxx.v2i or .iv2i"
        exit $1
    }

    $Ext = [string]$sFilePath.SubString($dotIdx)
    if ( $Ext -ne ".v2i" -And $Ext -ne ".iv2i")
	{
		Write-Host "The input file format must be either v2i or iv2i."
		exit $1;
	}
	
    #-----------------------------------------------------------------------------------------
	# Asking for network username and password if file is in network location.
	#-----------------------------------------------------------------------------------------
	if ($sFilePath.SubString(0,2) -eq "\\")
    {
        $bNetworkPath = $true
        $sUsrName = Read-Host "Enter the network username:"
        $NetSecurePwd = Read-Host -AsSecureString "Enter the network password:"
        $NetBSTR = [System.Runtime.InteropServices.Marshal]::SecureStringToBSTR($NetSecurePwd)            
        $sPwd = [System.Runtime.InteropServices.Marshal]::PtrToStringAuto($NetBSTR)
	}

    #-----------------------------------------------------------------------------------------
	# Step 2: Create a VProRecovery automation object.
	#-----------------------------------------------------------------------------------------
	
	$v2iAuto = New-Object -ComObject "Veritas.ProtectorAuto"
		
	#-----------------------------------------------------------------------------------------
	# Step 3: Connect to the local agent.
	#-----------------------------------------------------------------------------------------
	
    Write-Host "Connecting..."
	try
    {
	    $oNet = New-Object -ComObject "Wscript.Network"
        $v2iAuto.Connect($oNet.ComputerName)
    }
    catch
    {
        Write-Host "Failed to Connect with an Exception..."
        Write-Host $_.Exception.GetType().FullName -ForegroundColor Red
        Write-Host $_.Exception.Message -ForegroundColor Red
        Write-Host "Exiting..."
        exit $1
    }
	
	#----------------------------------------------------------------------------------------
	# Step 4: Define the location where the image will be verified from.
	#----------------------------------------------------------------------------------------
	$oLocation = CreateLocation $sFilePath $sUsrName $sPwd
    	
    #----------------------------------------------------------------------------------------
	# Step 5: Create a verify image job.
	#----------------------------------------------------------------------------------------
	$oVerifyJob = New-Object -ComObject "Veritas.VProRecovery.RestoreJob"
	$oVerifyJob.PreVerifyImage = $true                 # Verify the image
	$oVerifyJob.Source = $oLocation                    #Recovery point location
    
	#----------------------------------------------------------------------------------------
	# Step 6: Execute the Verify image job.
	#----------------------------------------------------------------------------------------
	
	Write-Host "Verifying..."
    Write-Host "--------------------------------------------------------------------------"

    do {
            try
            {
                $n = 0
                $v2iAuto.VerifyImageJob($oVerifyJob, $sImagePwd, $true)
            }
            catch
            {
		        #Write-Host $_.Exception.Message -ForegroundColor Red
		
		        $Arr = $_.Exception.Message -split ("file:")
		
                if ($arr.Count -gt 1)
                {
                    Write-Host "Validation failed for recovery points. Following is the list of corrupted images:" -ForegroundColor Red
		            for ($idx = 1; $idx -lt $Arr.Count ; $idx++) 
		            {
                        $startIdx = $Arr[$idx].IndexOf("\\")
                        $EndIdx = $Arr[$idx].IndexOf("v2i")
                        if ($bNetworkPath -eq $true)
                        {
                            $CorruptFileName = [string]$Arr[$idx].SubString($startIdx+2, $EndIdx+1)
                        }
                        else
                        {
                            
                            $CorruptFileName = [string]$Arr[$idx].SubString($startIdx+3, $EndIdx)
                        }
			            Write-Host $CorruptFileName -ForegroundColor Red
		            }
                    Write-Host "--------------------------------------------------------------------------"
                    Write-Host "Exiting..."
                    exit $1
		        }


		        if ($Arr.Count -eq 1)
		        {
			        if ( $_.Exception.Message.Contains("ERR_PASSWORD_REQUIRED") -eq $true)
			        {
                        if ($sImagePwd -eq $null)
                        {
                            $SecurePwd = Read-Host -AsSecureString "The image is password protected. Enter the password:"
                        }
                        else
                        {
                            $SecurePwd = Read-Host -AsSecureString "The password is incorrect. Enter the correct password:"
                        }
                        $BSTR = [System.Runtime.InteropServices.Marshal]::SecureStringToBSTR($SecurePwd)            
                        $sImagePwd = [System.Runtime.InteropServices.Marshal]::PtrToStringAuto($BSTR)  
                        $n = 1;
			        }
		
			        if ( $_.Exception.Message.Contains("ERROR_FILE_NOT_FOUND") -eq $true)
			        {
                        if ($bNetworkPath -eq $true)
                        {
				            Write-Host "The image path or network credentials are incorrect." -ForegroundColor Red
                        }
                        else
                        {
                            Write-Host "The image path is incorrect." -ForegroundColor Red
                        }
                        Write-Host "--------------------------------------------------------------------------"
                        Write-Host "Exiting..."
                        exit $1
			        }
		        }
            }

    } until ( $n -eq 0 )

 
    Write-Host "Recovery point validated successfully. No corrupted images found." -ForegroundColor Green
    Write-Host "--------------------------------------------------------------------------"