﻿#-----------------------------------------------------------------------------------------
#
# Veritas System Recovery 16 Automation
#
# Copyright (c) 2016 by Veritas Technologies LLC
# All Rights Reserved
#
# THIS SCRIPTING EXAMPLE IS PROVIDED IN CONJUNCTION WITH YOUR LICENSE FOR VERITAS 
# SYSTEM RECOVERY SOFTWARE PRODUCTS, AND MAY BE USED ONLY IN CONJUNCTION WITH 
# THAT SOFTWARE, UNDER THE TERMS OF THE END USER LICENSE AGREEMENT THAT ACCOMPANIED 
# THE SOFTWARE. THIS SCRIPTING EXAMPLE IS PROPRIETARY VERITAS PROPERTY.  YOU MAY NOT 
# COPY, DISTRIBUTE, LICENSE OR SUBLICENSE THE SCRIPTING DOCUMENTATION.
#
# 
#
#-----------------------------------------------------------------------------------------
##
#    Objects Used: ImageJob Location, LocationConstraints, Volume
#
#   Initialize the following variables with appropriate values:
#               1. sFilename to location of the .v2i file to be deleted.
#               2. oConstraints.ID to Job ID.
#               
#-----------------------------------------------------------------------------------------
	
	$v2iAuto = $null
	$oNet = $null
	$oLocation = $null
	$oImg = $null
	$oConstraints = $null
    $aVolumes = $null
    $oTempVol = $null
    $sFilename = $null
    
    #----------------------------------------------------------------------------------------------------
    # Function : CreateLocation
    # Input parameters :
    #                   a)  Path can be a local folder location or a network location.
    #                   b)  Domain user name if the path is a network location.
    #                   c)  password for authenticating to the network location.
    #
    # Return parameter: Location COM object containing all the information about the location
    #-----------------------------------------------------------------------------------------------------
    
    function CreateLocation 
    {
        param ([string]$Path, [string]$domainUser, [string]$password)
        
        $oLocation = $null
        $sepPos = $null 
        $dotPos = $null
        $Path = $Path -replace "/","\"
        
        # Parse the location.
        # Check if the location is a network location or a folder location.
        # If it is a network location, set the user name and password.
        if ($Path.SubString(0,2) -eq "\\")
        {
            $oLocation = New-Object -ComObject "Veritas.VProRecovery.NetworkLocation"
            $oLocation.NetworkUser = $domainUser
            $oLocation.NetworkPassword = $password
        }
        else 
        {
            $oLocation = New-Object -ComObject "Veritas.VProRecovery.FolderLocation"
        }
        
        $sepPos = $Path.LastIndexOf("\")
        $dotPos = $Path.LastIndexOf(".")

        $oLocation.Path = [string]$Path.SubString(0, $sepPos)
        $oLocation.FileSpec = [string]$Path.SubString($sepPos+1, $dotPos - $sepPos - 1)
        $oLocation.Extension = [string]$Path.SubString($dotPos)
        
        Write-Host "Path: " $oLocation.Path 
        Write-Host "FileSpecs: " $oLocation.FileSpec 
        Write-Host "Entension: "$oLocation.Extension
        
        return $oLocation
    }
    
    #-----------------------------------------------------------------------------------------
	# Step 1: Process command line arguments.
	#-----------------------------------------------------------------------------------------
	
	if ([int]$args.Length -gt 0) 
	{
		Write-Host "Usage: DeleteRecoveryPoints.ps1"
		exit $1;
	}
    
    #-----------------------------------------------------------------------------------------
	# Step 2: Create a VProRecovery automation object.
	#-----------------------------------------------------------------------------------------
	
	$v2iAuto = New-Object -ComObject "Veritas.ProtectorAuto"
		
	#-----------------------------------------------------------------------------------------
	# Step 3: Connect to the local agent.
	#-----------------------------------------------------------------------------------------
	
    Write-Host "Connecting..."
	try
    {
	    $oNet = New-Object -ComObject "Wscript.Network"
        $v2iAuto.Connect($oNet.ComputerName)
    }
    catch
    {
        Write-Host "Failed to Connect with an Exception..."
        Write-Host $_.Exception.GetType().FullName -ForegroundColor Red
        Write-Host $_.Exception.Message -ForegroundColor Red
        Write-Host "Exiting..."
        exit $1
    }

    #-----------------------------------------------------------------------------------------
    # Step 4: Define the location of the .v2i file to be deleted.
	#-----------------------------------------------------------------------------------------
	
	$sFilename = "G:\Veritas Backups\test\SystemBackup001.v2i"
	$oLocation = CreateLocation $sFilename "" ""
    
    #-----------------------------------------------------------------------------------------
	# Step 5:  Define Constraints object and set the parameters.
    #-------------------------------------------------------------------------
	#   ID - specify the job ID/image ID
	#   Context can be as below: 
	#     LOCATION_WRITE	= 0,
	#     LOCATION_READ	    = 1,
	#     LOCATION_WRITE_CREATE_DIR	=2
	#   Enter the image password. 
	#   Filename is the full path of the.v2i file location. It can be a local folder path or a network path.
	#   RequiredSizeMB is the size of the .v2i file in MB.
	#-----------------------------------------------------------------------------------------
    
    $oConstraints = New-Object -ComObject "Veritas.VProRecovery.LocationConstraints"
    $oConstraints.ID = "{B07F5842-DC58-43E5-A797-541520F9E20C}"
    $oConstraints.Context = 1 #$oConstraints.Constants.LOCATION_READ
    $oConstraints.Password = ""
    $oConstraints.Filename = $sFilename
    $oConstraints.RequiredSizeMB = 10245
    
    #-----------------------------------------------------------------------------------------
	# Step 6: Call the ImageForFile object.
	# Input parameters a) oLocation is the Location object.
	#                  b) oConstraints is the Constraints object.
	# Returns the Image object
	#-----------------------------------------------------------------------------------------
    
    Write-Host "Calling ImageForFile..."
    try
    {
        $oImg = $v2iAuto.ImageForFile($oLocation, $oConstraints )
    }
    catch
    {
        Write-Host "Image for File failed with an Exception..."
        Write-Host $_.Exception.GetType().FullName -ForegroundColor Red
        Write-Host $_.Exception.Message -ForegroundColor Red
        Write-Host "Exiting..."
        exit $1
    }
    
    #-----------------------------------------------------------------------------------------
	# Step 7:	
    #   Compare the drive letter of the Image object with the .v2i file location drive. 
	#   If true, remove the .v2i file.
	#-----------------------------------------------------------------------------------------

    #For Each vol in this
    $VolumeFound = $false
    foreach($oTempVol in $v2iAuto.Volumes($false))
    {
        if($oImg.Originaldriveletter -eq $oTempVol.GenericID)
        {
            Write-Host "Removing Image set..."
            try
            {
                $v2iAuto.RemoveImageSet($oTempVol.ID , $oImg.ID)
                $VolumeFound = $true
            }
            catch
            {
                Write-Host "Remove Image Set failed with an Exception..."
                Write-Host $_.Exception.GetType().FullName -ForegroundColor Red
                Write-Host $_.Exception.Message -ForegroundColor Red
                Write-Host "Exiting..."
                exit $1
            }
            break;
		}
    }
    
    if($VolumeFound -eq $false)
    {
        Write-Host "Image has a different drive letter than all the volumes on this system..."
        Write-Host "Cannot delete the recovery point..."
    }
    else
    {
        Write-Host "Recovery Point deleted successfully..."
    }