﻿#-----------------------------------------------------------------------------------------
#
# Veritas System Recovery 16 Automation
#
# Copyright (c) 2016 by Veritas Technologies LLC
# All Rights Reserved
#
# THIS SCRIPTING EXAMPLE IS PROVIDED IN CONJUNCTION WITH YOUR LICENSE FOR VERITAS 
# SYSTEM RECOVERY SOFTWARE PRODUCTS, AND MAY BE USED ONLY IN CONJUNCTION WITH 
# THAT SOFTWARE, UNDER THE TERMS OF THE END USER LICENSE AGREEMENT THAT ACCOMPANIED 
# THE SOFTWARE. THIS SCRIPTING EXAMPLE IS PROPRIETARY VERITAS PROPERTY.  YOU MAY NOT 
# COPY, DISTRIBUTE, LICENSE OR SUBLICENSE THE SCRIPTING DOCUMENTATION.
#
#-----------------------------------------------------------------------------------------
#   Example Script: Create an image of the system drive to a passed-in network folder.
#   Objects Used: ProtectorAuto, ImageJob, Volume, NetworkLocation, FolderLocation. 
#   Initialize the following variables with appropriate values:
#               1. sSrcNetLocation location of the .sv2i file. 
#               2. sDriveletter to a particular drive.
#-----------------------------------------------------------------------------------------
    
    $v2iAuto = $null
	$oNet = $null
	$oVolume = $null
	$oTempVol = $null
	$oSrcNetLocation = $null 
    $oDestNetLocation = $null
	$oImageJob = $null
	$oP2vSysInfo = $null
	$sDriveletter = $null
    $oP2vVolInfo = $null
    $sSrcNetLocation = $null
    
    #----------------------------------------------------------------------------------------------------
    # Function : CreateLocation
    # Input parameters :
    #                   a)  Path can be a local folder location or a network location.
    #                   b)  Domain user name if the path is a network location.
    #                   c)  password for authenticating to the network location.
    #
    # Return parameter: Location COM object containing all the information about the location
    #-----------------------------------------------------------------------------------------------------
    
    function CreateLocation 
    {
        param ([string]$Path, [string]$domainUser, [string]$password)
        
        $oLocation = $null
        $sepPos = $null 
        $dotPos = $null
        $Path = $Path -replace "/","\"
        
        # Parse the location.
        # Check if the location is a network location or a folder location.
        # If it is a network location, set the user name and password.
        if ($Path.SubString(0,2) -eq "\\")
        {
            $oLocation = New-Object -ComObject "Veritas.VProRecovery.NetworkLocation"
            $oLocation.NetworkUser = $domainUser
            $oLocation.NetworkPassword = $password
        }
        else 
        {
            $oLocation = New-Object -ComObject "Veritas.VProRecovery.FolderLocation"
        }
        
        $sepPos = $Path.LastIndexOf("\")
        $dotPos = $Path.LastIndexOf(".")

        $oLocation.Path = [string]$Path.SubString(0, $sepPos)
        $oLocation.FileSpec = [string]$Path.SubString($sepPos+1, $dotPos - $sepPos - 1)
        $oLocation.Extension = [string]$Path.SubString($dotPos)
        
        Write-Host "Path: " $oLocation.Path 
        Write-Host "FileSpecs: " $oLocation.FileSpec 
        Write-Host "Entension: "$oLocation.Extension
        
        return $oLocation
    }
    
    #---------------------------------------------------------------------------------------------
    # Function to get the system Sv2i location.
    # Depending on the location where the Veritas System Recovery is installed 
    # the location of system Sv2i varies. 
    # Use this function to automatically get the system Sv2i location.
    #---------------------------------------------------------------------------------------------

    function GetSystemSv2i
    {
        $COMMONAPPDATA = 40
        $sSystemSV2I = $null
        $sSystemSV2I = New-Object -ComObject "Shell.Application"
        $sNet = New-Object -ComObject "WScript.Network"
        $sSystemSV2I = $sSystemSV2I.NameSpace($COMMONAPPDATA).Self.Path + "\Veritas\Veritas System Recovery\" + $sNet.ComputerName + ".sv2i"
        Write-Host "System Sv2i Path : " $sSystemSV2I
        return $sSystemSV2I
    }
    
    #-----------------------------------------------------------------------------------------
	# Step 1: Process command line arguments.
	#-----------------------------------------------------------------------------------------
	
	if ([int]$args.Length -gt 0) 
	{
		Write-Host "Usage: CreateVirtualMachineFromSv2i.ps1"
		exit $1;
	}
    
    $sDriveletter = "C:\"
	
    #-----------------------------------------------------------------------------------------
	# Step 2: Create a VProRecovery automation object.
	#-----------------------------------------------------------------------------------------
	
	$v2iAuto = New-Object -ComObject "Veritas.ProtectorAuto"
		
	#-----------------------------------------------------------------------------------------
	# Step 3: Connect to the local agent.
	#-----------------------------------------------------------------------------------------
	
    Write-Host "Connecting..."
	try
    {
	    $oNet = New-Object -ComObject "Wscript.Network"
        $v2iAuto.Connect($oNet.ComputerName)
    }
    catch
    {
        Write-Host "Failed to Connect with an Exception..."
        Write-Host $_.Exception.GetType().FullName -ForegroundColor Red
        Write-Host $_.Exception.Message -ForegroundColor Red
        Write-Host "Exiting..."
        exit $1
    }
    
    #-----------------------------------------------------------------------------------------
	# Step 4: Define the location.
	#          User can set the location of the .sv2i file. 
	#          else use
	#               GetSystemSv2i() - which automatically gets the system sv2i location. 
	#-----------------------------------------------------------------------------------------
	
	$sSrcNetLocation = "F:\Veritas Backups\Images\HONSHU-MS.sv2i"
	#$sSrcNetLocation = GetSystemSv2i    #If the user want to get the System Sv2i location
	$oSrcNetLocation = CreateLocation "F:\Veritas Backups\Images\HONSHU-MS.sv2i" "" ""
	#$oDestNetLocation = CreateLocation "F:\Veritas Backups\test\A2SYMMD03449.sv2i" "" ""
    
	#-----------------------------------------------------------------------------------------
	# Step 5: Call LoadSV2i to update the oP2vSysInfo object.
	#-----------------------------------------------------------------------------------------
    Write-Host "Lodaing Sv2i File..."
    try
    {
        $oP2vSysInfo =  $v2iAuto.LoadSV2i($oSrcNetLocation )
    }
    catch
    {
        Write-Host "Failed to load the Sv2i file with an Exception..."
        Write-Host $_.Exception.GetType().FullName -ForegroundColor Red
        Write-Host $_.Exception.Message -ForegroundColor Red
        Write-Host "Exiting..."
        exit $1
    }
    Write-Host "Loading Sv2i File Successful..."

    #-----------------------------------------------------------------------------------------
    # oP2vSysInfo object will be updated for the following:
    #
	#  oP2vSysInfo
	#  Parameters are used in the following order:
	#               SourceLocation
	#               DestinationLocation
	#               DestinationConversionType  
	#                     P2V_TYPE_V2I = 0,	// convert to .v2i
	#                     P2V_TYPE_VMDK,		// convert to .vmdk
	#                     P2V_TYPE_VHD,		// convert to .vhd
	#                     P2V_TYPE_D2I,		// convert to .d2i
	#               MemorySize
	#               Set OSType , RunJobDisplayText ,SysPrep
	#-----------------------------------------------------------------------------------------
    
    #	$oP2vSysInfo.SourceLocation = oSrcNetLocation
    #   $oP2vSysInfo.ComputerName = "MyVMMachine" 
    #   $oP2vSysInfo.DestinationConversionType =  1 #oP2vVolInfo.Constants.P2V_TYPE_VMDK
    #   $oP2vSysInfo.MemorySize = 1199
    #   $oP2vSysInfo.OSType = "Windows XP"
    #   $oP2vSysInfo.RunJobDisplayText = "Virtual Conversion of My local Machine"
    #   $oP2vSysInfo.SysPrep = True 
    #   $oP2vSysInfo.OneDiskPerVolume = 0
    #   $oP2vSysInfo.Enabled = True
    #   $oP2vSysInfo.VolumesToConvert  = Array(sDriveletter)
    #     
    #-----------------------------------------------------------------------------------------
    
    #-----------------------------------------------------------------------------------------
	# Step 6: Set additional parameter of the oP2vSysInfo object, if needed.
	#         Call ConvertSystem to convert it to virtual file (.vmdk or .vhd).
	#-----------------------------------------------------------------------------------------
	
    #$oP2vSysInfo.ComputerName = "MyVMMachine" 
   	#$oP2vSysInfo.DestinationLocation = oDestNetLocation
    #$oP2vSysInfo.DestinationConversionType =  1 #oP2vVolInfo.Constants.P2V_TYPE_VMDK

    Write-Host "Calling ConvertSystem..."
    try
    {
        $v2iAuto.ConvertSystem($oP2vSysInfo)
    }
    catch
    {
        Write-Host "Failed in Convert System with an Exception..."
        Write-Host $_.Exception.GetType().FullName -ForegroundColor Red
        Write-Host $_.Exception.Message -ForegroundColor Red
        Write-Host "Exiting..."
        exit $1
    }
    Write-Host "Convert System successfully..."