﻿#-----------------------------------------------------------------------------------------
#
# Veritas System Recovery 16 Automation
#
# Copyright (c) 2016 by Veritas Technologies LLC
# All Rights Reserved
#
# THIS SCRIPTING EXAMPLE IS PROVIDED IN CONJUNCTION WITH YOUR LICENSE FOR VERITAS 
# SYSTEM RECOVERY SOFTWARE PRODUCTS, AND MAY BE USED ONLY IN CONJUNCTION WITH 
# THAT SOFTWARE, UNDER THE TERMS OF THE END USER LICENSE AGREEMENT THAT ACCOMPANIED 
# THE SOFTWARE. THIS SCRIPTING EXAMPLE IS PROPRIETARY VERITAS PROPERTY.  YOU MAY NOT 
# COPY, DISTRIBUTE, LICENSE OR SUBLICENSE THE SCRIPTING DOCUMENTATION.
#
#-----------------------------------------------------------------------------------------
#   Example Script: Create an image of the system drive to a passed-in network folder.
#   Objects Used: ProtectorAuto, ImageJob, Volume, NetworkLocation , FolderLocation , oP2vSysInfo
#                   oVMDKDiskOption , oP2vDiskInfo , oP2vVolInfo.
#   Initialize the following variables with appropriate values :
#               1. oSrcNetLocation and oDestNetLocation to location 
#               2. sDriveletter to a particular drive
#-----------------------------------------------------------------------------------------
    
    $v2iAuto = $null
	$oNet = $null
	$oVolume = $null
	$oTempVol = $null
	$oSrcNetLocation = $null 
    $oDestNetLocation = $null
	$oImageJob = $null
	$sFolder = $null
	$oP2vSysInfo = $null
	$sDriveletter = $null
	$oVMDKDiskOption = $null
    $oP2vDiskInfo = $null
    $oP2vVolInfo = $null
    
    #-----------------------------------------------------------------------------------------
	# Step 1: Process command line arguments.
	#-----------------------------------------------------------------------------------------
	
	if ([int]$args.Length -gt 0) 
	{
		Write-Host "Usage: CreateVirtual.ps1"
		exit $1;
	}
    
    $sDriveletter = "C:\"
	$sFolder = "F:\Veritas Backups\Images"     #This value is [UNC Network Path] 
	
    #-----------------------------------------------------------------------------------------
	# Step 2: Create a VProRecovery automation object.
	#-----------------------------------------------------------------------------------------
	
	$v2iAuto = New-Object -ComObject "Veritas.ProtectorAuto"
		
	#-----------------------------------------------------------------------------------------
	# Step 3: Connect to the local agent.
	#-----------------------------------------------------------------------------------------
	
    Write-Host "Connecting..."
	try
    {
	    $oNet = New-Object -ComObject "Wscript.Network"
        $v2iAuto.Connect($oNet.ComputerName)
    }
    catch
    {
        Write-Host "Failed to Connect with an Exception..."
        Write-Host $_.Exception.GetType().FullName -ForegroundColor Red
        Write-Host $_.Exception.Message -ForegroundColor Red
        Write-Host "Exiting..."
        exit $1
    }
    
    #-----------------------------------------------------------------------------------------
	# Step 4: Define the network location for saving the image. 
	#-----------------------------------------------------------------------------------------

    $oSrcNetLocation = New-Object -ComObject "Veritas.VProRecovery.FolderLocation"
	$oSrcNetLocation.Path = $sFolder
	$oSrcNetLocation.FileSpec = "SystemBackup"
	$oSrcNetLocation.Extension = ".v2i"
	#oNetLocation.NetworkUser = "DOMAIN\UserName"
	#oNetLocation.NetworkPassword = "mypassword"
	$oDestNetLocation = New-Object -ComObject "Veritas.VProRecovery.FolderLocation"
	$oDestNetLocation.Path = $sFolder
	$oDestNetLocation.FileSpec = "SystemBackup"
	$oDestNetLocation.Extension = ".vmdk"
    
    #-----------------------------------------------------------------------------------------
	# Step 5: Find the volume to image (the system volume in this case).
	#-----------------------------------------------------------------------------------------
	
    foreach($oTempVol in $v2iAuto.Volumes($false))
    {
		If ($oTempVol.System -eq $true)
        {
			$oVolume = $oTempVol
			break;
		}
	}

	if($oVolume -eq $null)
    {
		Write-Host "CreateVirtual.ps1: Cannot find requested volume..."
        Write-Host "Exiting..."
        exit $1
    }
    
    #-------------------------------------------------------------------------------------------
    #  Step 6: Create a manual image job that automatically removes itself after running.
	#  Parameters
	#           IncrementalSupport is false
	#           Specify the Display Name and Description
	#           Compression types
	#               IMAGE_COMPRESSION_LOW	
	#               IMAGE_COMPRESSION_MEDIUM	
	#               IMAGE_COMPRESSION_HIGH	
	#               IMAGE_COMPRESSION_NEWMEDIUM 
	#           RunOnce is true
    #--------------------------------------------------------------------------------------------

    $oImageJob = New-Object -ComObject "Veritas.VProRecovery.ImageJob"
	$oImageJob.IncrementalSupport = $false
	$oImageJob.DisplayName = "Recovery point of " + $oVolume.DisplayName
	$oImageJob.Description = "Test recovery point of the system volume."
	$oImageJob.Compression = $oImageJob.Constants.ImageCompressionLow
	$oImageJob.Reason = $oImageJob.Constants.ImageReasonManual
	$oImageJob.Volumes = [Array]$oVolume.ID
	$oImageJob.Location($oVolume.ID) = $oSrcNetLocation
	$oImageJob.RunOnce = $true
    
    #-----------------------------------------------------------------------------------------
	# Step 7: Add the image job to the list of jobs.
	#-----------------------------------------------------------------------------------------

    Write-Host "Adding Image Job..."
    try
    {
        $v2iAuto.AddImageJob($oImageJob)
    }
    catch
    {
        Write-Host "Failed to Add Image Job with an Exception..."
        Write-Host $_.Exception.GetType().FullName -ForegroundColor Red
        Write-Host $_.Exception.Message -ForegroundColor Red
        Write-Host "Exiting..."
        exit $1
    }
	Write-Host "Image Job added successfully."
    
    #-----------------------------------------------------------------------------------------
	# Step 8: Run the image job immediately.
	#-----------------------------------------------------------------------------------------
    
	Write-Host "Creating image..."
    try
    {
	    $v2iAuto.DoImageJob($oImageJob.ID, $oImageJob.Constants.ImageTypeFull)
    }
    catch
    {
        Write-Host "Failed to Create Image with an Exception..."
        Write-Host $_.Exception.GetType().FullName -ForegroundColor Red
        Write-Host $_.Exception.Message -ForegroundColor Red
        Write-Host "Exiting..."
        exit $1
    }
	Write-Host "Image created successfully..."
    
    #-----------------------------------------------------------------------------------------
    #   Step 9: Define parameters for oP2vSysInfo.
	#   Parameters used in the following order
	#               SourceLocation
	#               DestinationLocation
	#               DestinationConversionType  
	#                     P2V_TYPE_V2I = 0,	// convert to .v2i
	#                     P2V_TYPE_VMDK,		// convert to .vmdk
	#                     P2V_TYPE_VHD,		// convert to .vhd
	#                     P2V_TYPE_D2I,		// convert to .d2i
	#               MemorySize
	#               Set OSType , RunJobDisplayText ,SysPrep is true /false
	#               Set OneDiskPerVolume to zero
	#               Enabled is true/false , SysPrep
    #-----------------------------------------------------------------------------------------
    
    $oP2vSysInfo = New-Object -ComObject "Veritas.VProRecovery.P2vSystemInfo"
	$oP2vSysInfo.SourceLocation = $oSrcNetLocation
	$oP2vSysInfo.DestinationLocation = $oDestNetLocation
    $oP2vSysInfo.ComputerName = "MyVMMachine" 
    $oP2vSysInfo.DestinationConversionType = 1 #$oP2vVolInfo.Constants.P2V_TYPE_VMDK
    $oP2vSysInfo.MemorySize = 1199
    $oP2vSysInfo.OSType = "Windows XP"
    $oP2vSysInfo.RunJobDisplayText = "Virtual Conversion of My local Machine"
    $oP2vSysInfo.SysPrep = $true 
    $oP2vSysInfo.OneDiskPerVolume = 0
    $oP2vSysInfo.Enabled = $true
    $oP2vSysInfo.VolumesToConvert  = [Array]$sDriveletter
    
    #-----------------------------------------------------------------------------------------
    #  Step 10: Define parameters for  oVMDKDiskOption
	#  Parameters used in the following order
	#  VMDKDiskOption     VMDK_DISKTYPE_SPAN = 0,
	#                     VMDK_DISKTYPE_MONOLITHIC,
	#                     VMDK_DISKTYPE_SPARSE,
	#  TargetProduct
    #                    VMDK_TARGET_GSX = 0,
    #                    VMDK_TARGET_ESX,
    #                    VMDK_TARGET_WORKSTATION4,
    #                    VMDK_TARGET_WORKSTATION5,
    #                    VMDK_TARGET_WORKSTATION6,
    #                    VMDK_TARGET_SERVER1
    #-----------------------------------------------------------------------------------------
    
    $oVMDKDiskOption = New-Object -ComObject "Veritas.VProRecovery.VmdkDiskOptions"
    $oVMDKDiskOption.DiskType = 1 #$oVMDKDiskOption.Constants.VMDK_DISKTYPE_MONOLITHIC
    $oVMDKDiskOption.TargetProduct = 5 #$oVMDKDiskOption.Constants.VMDK_TARGET_WORKSTATION6
    $oVMDKDiskOption.SpanSize = $false
    $oVMDKDiskOption.RemoveIntermediateFiles = $true
    
    #------------------------------------------------------------------------------------------
    #  Step 11: Define parameters for oP2vDiskInfo 
	#  Parameters used in the following order
	#  P2vDiskInfo
	#       Set the following
	#               Destination location,
	#               Size in MB 
	#               CreateVirtualDisk is true if user wants to create virtual disk else false
	#               Options	set to oVMDKDiskOption object
    #-------------------------------------------------------------------------------------------
    
    $oP2vDiskInfo = New-Object -ComObject "Veritas.VProRecovery.P2vDiskInfo"
    $oP2vDiskInfo.Location = $oDestNetLocation
    $oP2vDiskInfo.DeviceNumber = 0
    $oP2vDiskInfo.SizeMB = 10245
    $oP2vDiskInfo.CreateVirtualDisk = $true
    $oP2vDiskInfo.Options = $oVMDKDiskOption
    
    #After gettin the oP2vDiskInfo set the oP2vSysInfo.Disks parameter
    $oP2vSysInfo.Disks = [Array]$oP2vDiskInfo
    $oP2vSysInfo.Options = $oVMDKDiskOption
    
    #-------------------------------------------------------------------------------------------
    #  Step 12: Define parameters for  oP2vVolInfo 
	#  Parameters 
	#  Set the parameters in the following order
	#               DestinationConversionType  
	#                     P2V_TYPE_V2I = 0,	// convert to .v2i
	#                     P2V_TYPE_VMDK,	// convert to .vmdk
	#                     P2V_TYPE_VHD,		// convert to .vhd
	#               DriveLetter is the drive letter
	#               FreeSpaceMB 
	#               IncludeInVirtualDisk is true/false
	#               IsActive is true/false
	#               IsLogical is true/false
	#               IsSystemVolume is true/false
	#               OffsetOnMedia set it to zero
	#               SizeMB in MB
	#               SourceLocation is the source location
    #-------------------------------------------------------------------------------------------
    
    $oP2vVolInfo = New-Object -ComObject "Veritas.VProRecovery.P2vVolumeInfo"
    $oP2vVolInfo.DestinationConversionType = 1 #$oP2vVolInfo.Constants.P2V_TYPE_VMDK
    $oP2vVolInfo.DriveLetter = $oVolume.GenericID  
    $oP2vVolInfo.FreeSpaceMB = $oVolume.FreeSpaceMB
    $oP2vVolInfo.IncludeInVirtualDisk = $true
    $oP2vVolInfo.IsActive = $true
    $oP2vVolInfo.IsLogical = $true
    $oP2vVolInfo.IsSystemVolume = $true
    $oP2vVolInfo.OffsetOnMedia = 0 #$oVolume.OffsetOnMedia  
    $oP2vVolInfo.SizeMB  = $oVolume.SizeMB
    $oP2vVolInfo.VolumeId = $oVolume.ID
    $oP2vVolInfo.GenericID = $oVolume.GenericID
    $oP2vVolInfo.SourceLocation = $oSrcNetLocation    
    $oP2vDiskInfo.Volumes =  [Array]$oP2vVolInfo
    
    #-------------------------------------------------------------------------------------------
    # Step 13: Call the ConvertSystem object to convert to virtual disk.
    #-------------------------------------------------------------------------------------------
    
    Write-Host "Calling ConvertSystem..."
    try
    {
        $v2iAuto.ConvertSystem($oP2vSysInfo)
    }
    catch
    {
        Write-Host "Failed to Convert to virtual disk with an Exception..."
        Write-Host $_.Exception.GetType().FullName -ForegroundColor Red
        Write-Host $_.Exception.Message -ForegroundColor Red
        Write-Host "Exiting..."
        exit $1
    }
    Write-Host "Converted to virtual disk successfully..."