﻿#-----------------------------------------------------------------------------------------
#
# Veritas System Recovery 16 Automation
#
# Copyright (c) 2016 by Veritas Technologies LLC
# All Rights Reserved
#
# THIS SCRIPTING EXAMPLE IS PROVIDED IN CONJUNCTION WITH YOUR LICENSE FOR VERITAS 
# SYSTEM RECOVERY SOFTWARE PRODUCTS, AND MAY BE USED ONLY IN CONJUNCTION WITH 
# THAT SOFTWARE, UNDER THE TERMS OF THE END USER LICENSE AGREEMENT THAT ACCOMPANIED 
# THE SOFTWARE. THIS SCRIPTING EXAMPLE IS PROPRIETARY VERITAS PROPERTY.  YOU MAY NOT 
# COPY, DISTRIBUTE, LICENSE OR SUBLICENSE THE SCRIPTING DOCUMENTATION.
#
#-----------------------------------------------------------------------------------------
#
#  Example Script: Create a base image of the system drive that can be used later to create incremental images.
#  Objects Used: ProtectorAuto, ImageJob, Volume, NetworkLocation.
#  Initialize the following variables with appropriate values:
#               1. sFolder
#               2. StrVolume  
#-----------------------------------------------------------------------------------------
    
    $v2iAuto = $null
	$oNet = $null
	$oVolume = $null
	$oTempVol = $null
	$oNetLocation = $null
	$oImageJob = $null
	$sFolder = $null
	$oFileSysObj = $null
	$oFile = $null
	$IsUser = $null
    $IsPwd = $null
	$bChooseVoulme = $null
    $StrVolume = $null
    $bContinue = $null
    $DomainUser = $null
    $Password = $null
    
    #----------------------------------------------------------------------------------------------------
    # Function : CreateLocation
    # Input parameters :
    #                   a)  Path can be a local folder location or a network location.
    #                   b)  Domain user name if the path is a network location.
    #                   c)  password for authenticating to the network location.
    #
    # Return parameter: Location COM object containing all the information about the location
    #-----------------------------------------------------------------------------------------------------
    
    [reflection.assembly]::loadwithpartialname("System.Windows.Forms") | Out-Null 
    
    function CreateLocation 
    {
        param ([string]$Path, [string]$domainUser, [string]$password)
        
        $oLocation = $null
        $sepPos = $null 
        $dotPos = $null
        $Path = $Path -replace "/","\"
        
        # Parse the location.
        # Check if the location is a network location or a folder location.
        # If it is a network location, set the user name and password.
        if ($Path.SubString(0,2) -eq "\\")
        {
            $oLocation = New-Object -ComObject "Veritas.VProRecovery.NetworkLocation"
            $oLocation.NetworkUser = $domainUser
            $oLocation.NetworkPassword = $password
        }
        else 
        {
            $oLocation = New-Object -ComObject "Veritas.VProRecovery.FolderLocation"
        }
        
        $sepPos = $Path.LastIndexOf("\")
        $dotPos = $Path.LastIndexOf(".")

        $oLocation.Path = [string]$Path.SubString(0, $sepPos)
        $oLocation.FileSpec = [string]$Path.SubString($sepPos+1, $dotPos - $sepPos - 1)
        $oLocation.Extension = [string]$Path.SubString($dotPos)
        
        Write-Host "Path: " $oLocation.Path 
        Write-Host "FileSpecs: " $oLocation.FileSpec 
        Write-Host "Entension: "$oLocation.Extension
        
        return $oLocation
    }
    
    #---------------------------------------------------------------------------------------
    # Function to Find the volume
    #---------------------------------------------------------------------------------------

    function FindVolume
    {
        param ([bool]$bChooseVolume, [string]$Volume)
        $strVolume = $null

        if($bChooseVolume -eq $false)
        {
            foreach($oTempVol in $v2iAuto.Volumes($false))
            {
		        if ($oTempVol.System -eq $true)
                {
			        $strVolume = $oTempVol
                    break;
                }
	        }
        }
        else
        {
  	        foreach ($oTempVol in $v2iAuto.Volumes($false))
            {
		        if($oTempVol.MountPoint -eq $Volume)
                {
			        $strVolume = $oTempVol
			        break;
		        }
	        }
        }
        return $strVolume     
    }
    
    #-----------------------------------------------------------------------------------------
	# Step 1: Process command line arguments.
	#-----------------------------------------------------------------------------------------
	
	if ([int]$args.Length -gt 0) 
	{
		Write-Host "Usage: CreateBaseImageNow.ps1"
		exit $1;
	}
	
	#-----------------------------------------------------------------------------------------
	# Step 2: Create a VProRecovery automation object.
	#-----------------------------------------------------------------------------------------
	
	$v2iAuto = New-Object -ComObject "Veritas.ProtectorAuto"
		
	#-----------------------------------------------------------------------------------------
	# Step 3: Connect to the local agent.
	#-----------------------------------------------------------------------------------------
	
    Write-Host "Connecting..."
	try
    {
	    $oNet = New-Object -ComObject "Wscript.Network"
        $v2iAuto.Connect($oNet.ComputerName)
    }
    catch
    {
        Write-Host "Failed to Connect with an Exception..."
        Write-Host $_.Exception.GetType().FullName -ForegroundColor Red
        Write-Host $_.Exception.Message -ForegroundColor Red
        Write-Host "Exiting..."
        exit $1
    }
    
    #-----------------------------------------------------------------------------------------------------------
	# Step 4: 
    #   By default, the image is stored in local system folder location.
	#   You can change the storage location to network by setting the sFolder varible as explained below:
	#   sFolder=\\Ipadress or machine name\share folder name. Also set the network user name as explained below:
	#   DomainUser="Domain\Username"
    #   Password="MyPassword"       
    #------------------------------------------------------------------------------------------------------------
    
    $sFolder = "E:\Images\base.v2i"
    $DomainUser = "" 
    $Password = "" 

	#--------------------------------------------------------------------------------------------
	# By default, we will create an image of the system volume.
	# If you want to create an image of a specific volume, specify so as explained below: 
	# varible bChooseVolume to non zero value and set the variable StrVolume with specific volume.
	# Example: StrVolume="D:\"           
	#--------------------------------------------------------------------------------------------
	
	$bChooseVoulme = $true
	$StrVolume = "F:\" 
    
    #-------------------------------------------------------------------------------------------
	# Step 5: Define the network location\folder location for storing the image. 
	# If it is an network location pass Domain\user name and password for 2nd and 3rd parameter.    
	#-------------------------------------------------------------------------------------------
	
    $oNetLocation = CreateLocation $sFolder $DomainUser $Password
	
	#-----------------------------------------------------------------------------------------
	# Step 6: By default, we will find the system volume to image.
	#-----------------------------------------------------------------------------------------
	
    $oVolume = FindVolume $bChooseVoulme $StrVolume
	
	if($oVolume -eq $null)
    {
		Write-Host "CreateBaseImageNow.ps1. Cannot find requested volume..."
        Write-Host "Exiting..."
        exit $1
    }
    
    #-----------------------------------------------------------------------------------------
	# Step 7: Create a base image job.
	# Parameters to set
	#          Set the Type, Reason 
	#          IncrementalSupport is true/false
	#          RunOnce is true/false
	#          Description - specify a Description for the image job
	#          DisplayName - specify a display name for the image job
	#          Compression types
	#                        IMAGE_COMPRESSION_LOW	
	#                        IMAGE_COMPRESSION_MEDIUM	
	#                        IMAGE_COMPRESSION_HIGH	
	#                        IMAGE_COMPRESSION_NEWMEDIUM           
	#-----------------------------------------------------------------------------------------
	
	$oImageJob = New-Object -ComObject "Veritas.VProRecovery.ImageJob"
	$oImageJob.Type = $oImageJob.Constants.ImageTypeBase
	$oImageJob.Reason = $oImageJob.Constants.ImageReasonManual
	$oImageJob.IncrementalSupport = $true
    $oImageJob.SkipBadSectors = $true
	$oImageJob.RunOnce = $false
	$oImageJob.DisplayName = "Recovery point of " + $oVolume.DisplayName
	$oImageJob.Description = "Recovery point of the system volume."
	$oImageJob.Compression = $oImageJob.Constants.ImageCompressionLow
	$oImageJob.Volumes = [Array]$oVolume.ID
	$oImageJob.Location($oVolume.ID) = $oNetLocation    # location for this volume#s image
    
    #-----------------------------------------------------------------------------------------
	# Step 8: 	Add the image job to the job list.
	#-----------------------------------------------------------------------------------------
	
	Write-Host "Creating Recovery point..."
    try
    {
	    $V2iAuto.AddImageJob($oImageJob)
    }
    catch
    {
        Write-Host "Failed to Add Recovery point with an Exception..."
        Write-Host $_.Exception.GetType().FullName -ForegroundColor Red
        Write-Host $_.Exception.Message -ForegroundColor Red
        Write-Host "Exiting..."
        exit $1
    }
	Write-Host "Adding Recovery point successful..."
	
	#Start taking recovery pont of speicifed drive..
    $result = [System.Windows.Forms.MessageBox]::Show(("This will start taking recovery point of " + $oVolume.DisplayName + "`r`nDo you want to continue?"),
                "Run recovery job",
                [System.Windows.Forms.MessageBoxButtons]::YesNo,
                [System.Windows.Forms.MessageBoxIcon]::Question,
                [System.Windows.Forms.MessageBoxDefaultButton]::Button1)
    
    if($result -eq [System.Windows.Forms.DialogResult]::Yes) 
    {
        Write-Host "Running Job..."
        try
        {
    	    $v2iAuto.DoImageJob($oImageJob.ID, $oImageJob.Constants.ImageTypeBase)
        }
        catch
        {
            Write-Host "Failed to run the Job with an Exception..."
            Write-Host $_.Exception.GetType().FullName -ForegroundColor Red
            Write-Host $_.Exception.Message -ForegroundColor Red
            Write-Host "Exiting..."
            exit $1
        }
        Write-Host "Job " $oImageJob.ID "completed successfully."
    }
