﻿#-----------------------------------------------------------------------------------------
#
# Veritas System Recovery 16 Automation
#
# Copyright (c) 2016 by Veritas Technologies LLC
# All Rights Reserved
#
# THIS SCRIPTING EXAMPLE IS PROVIDED IN CONJUNCTION WITH YOUR LICENSE FOR VERITAS 
# SYSTEM RECOVERY SOFTWARE PRODUCTS, AND MAY BE USED ONLY IN CONJUNCTION WITH 
# THAT SOFTWARE, UNDER THE TERMS OF THE END USER LICENSE AGREEMENT THAT ACCOMPANIED 
# THE SOFTWARE. THIS SCRIPTING EXAMPLE IS PROPRIETARY VERITAS PROPERTY.  YOU MAY NOT 
# COPY, DISTRIBUTE, LICENSE OR SUBLICENSE THE SCRIPTING DOCUMENTATION.
#
#-----------------------------------------------------------------------------------------
#
#  Example Script: Copy a .v2i file to a particular destination.
#  Objects Used: ProtectorAuto object, NetworkLocation object, oP2vVolInfo object
#  Initialize the following variables with appropriate values:
#               1. sFolder with source folder location.
#               2. sDestFolder with destination folder location.
#-----------------------------------------------------------------------------------------
    
	$v2iAuto = $null
	$oNet = $null
	$oVolume = $null
	$oTempVol = $null
	$oSrcNetLocation = $null 
    $oDestNetLocation = $null
	$oImageJob = $null
	$sFolder = $null
	$sDestFolder = $null
    $oP2vVolInfo = $null
    
     #----------------------------------------------------------------------------------------------------
    # Function : CreateLocation
    # Input parameters :
    #                   a)  Path can be a local folder location or a network location.
    #                   b)  Domain user name if the path is a network location.
    #                   c)  password for authenticating to the network location.
    #
    # Return parameter: Location COM object containing all the information about the location
    #-----------------------------------------------------------------------------------------------------
    
    function CreateLocation 
    {
        param ([string]$Path, [string]$domainUser, [string]$password)
    	
        $oLocation = $null
        $sepPos = $null 
        $dotPos = $null
        $Path = $Path -replace "/","\"
        
        # Parse the location.
        # Check if the location is a network location or a folder location.
        # If it is a network location, set the user name and password.
        if ($Path.SubString(0,2) -eq "\\")
        {
            $oLocation = New-Object -ComObject "Veritas.VProRecovery.NetworkLocation"
            $oLocation.NetworkUser = $domainUser
            $oLocation.NetworkPassword = $password
        }
        else 
        {
            $oLocation = New-Object -ComObject "Veritas.VProRecovery.FolderLocation"
        }
        
        $sepPos = $Path.LastIndexOf("\")
        $dotPos = $Path.LastIndexOf(".")

        $oLocation.Path = $Path.SubString(0, $sepPos)
        $oLocation.FileSpec = $Path.SubString($sepPos+1, $dotPos - $sepPos - 1)
        $oLocation.Extension = $Path.SubString($dotPos)

        # Return the oLocation object
        return $oLocation
    }
    
    #-----------------------------------------------------------------------------------------
	# Step 1: Process command line arguments.
	#-----------------------------------------------------------------------------------------
	
	if ([int]$args.Length -gt 0) 
	{
		Write-Host "Usage: CopyToDestination.ps1"
		exit $1;
	}
	
	#-----------------------------------------------------------------------------------------
	# Step 2: Create a VProRecovery automation object.
	#-----------------------------------------------------------------------------------------
	
	$v2iAuto = New-Object -ComObject "Veritas.ProtectorAuto"
		
	#-----------------------------------------------------------------------------------------
	# Step 3: Connect to the local agent.
	#-----------------------------------------------------------------------------------------
	
    Write-Host "Connecting..."
	try
    {
	    $oNet = New-Object -ComObject "Wscript.Network"
        $v2iAuto.Connect($oNet.ComputerName)
    }
    catch
    {
        Write-Host "Failed to Connect with an Exception..."
        Write-Host $_.Exception.GetType().FullName -ForegroundColor Red
        Write-Host $_.Exception.Message -ForegroundColor Red
        Write-Host "Exiting..."
        exit $1
    }
    
    #---------------------------------------------------------------------------------------
    # Step 4: Set the source location and the destination location.
    #---------------------------------------------------------------------------------------
	
    $sFolder = "G:\Veritas Backups\test\SystemBackup002.v2i"
    $sDestFolder = "E:\Images\SystemBackup002.v2i"
    
    #---------------------------------------------------------------------------------------
	# Step 5: Define the network/folder location for saving the image. 
	#---------------------------------------------------------------------------------------
	
	$oSrcNetLocation = CreateLocation $sFolder "" ""
	$oDestNetLocation = CreateLocation $sDestFolder "" ""
    
    #---------------------------------------------------------------------------------------
	# Step 6: Find the volume to image (the system volume in this case).
	#---------------------------------------------------------------------------------------
	
    foreach ($oTempVol in $v2iAuto.Volumes($false))
    {
		if ($oTempVol.System -eq $true)
        {
			$oVolume = $oTempVol
			break;
		}
    }
	If ($oVolume -eq $null)
    {
		Write-Host "CopyToDestination.ps1: Cannot find requested volume..."
        Write-Host "Exiting..."
        exit $1
    }
    
    #---------------------------------------------------------------------------------------------------------------
    # Step 7: oP2vVolInfo object.
	#           This oP2vVolInfo object is used to copy the .v2i file from the source to the destination location.
    # Parameters 
	#               DestinationConversionType  
	#                     P2V_TYPE_V2I = 0,	// convert to .v2i
	#                     P2V_TYPE_VMDK,		// convert to .vmdk
	#                     P2V_TYPE_VHD,		// convert to .vhd
	#                     P2V_TYPE_D2I,		// convert to .d2i
	#               Specify the drive letter
	#               Specify the free space in MB
	#               Set IsActive
	#               Set IsLogical to true/false
	#               Set IsSystemVolume to true/false
    #               Set OffsetOnMedia to zero
    #               Set Size in MB
	#               SourceLocation is the source location of .v2i file.
	#               DestinationLocation is the destination location, where the.v2i file will be copied.
	#-----------------------------------------------------------------------------------------------------------------
	
    $oP2vVolInfo = New-Object -ComObject "Veritas.VProRecovery.P2vVolumeInfo"
    $oP2vVolInfo.DestinationConversionType = 1 #$oP2vVolInfo.Constants.P2V_TYPE_VMDK
    $oP2vVolInfo.DriveLetter = $oVolume.GenericID  
    $oP2vVolInfo.FreeSpaceMB = $oVolume.FreeSpaceMB
    $oP2vVolInfo.IncludeInVirtualDisk = $true
    $oP2vVolInfo.IsActive = $true                 
    $oP2vVolInfo.IsLogical = $true                
    $oP2vVolInfo.IsSystemVolume = $true           
    $oP2vVolInfo.OffsetOnMedia = 0               
    $oP2vVolInfo.SizeMB  = $oVolume.SizeMB
    $oP2vVolInfo.VolumeId = $oVolume.ID
    $oP2vVolInfo.GenericID = $oVolume.GenericID
    $oP2vVolInfo.SourceLocation = $oSrcNetLocation
    $oP2vVolInfo.DestinationLocation = $oDestNetLocation
    
    #-----------------------------------------------------------------------------------------------------
	# Step 8: Call Convert Volume.
	#   Inputs - Array of P2V volume info
	#-----------------------------------------------------------------------------------------------------
	Write-Host "Copying to destination..."
    try
    {
        $v2iAuto.ConvertVolume([Array]$oP2vVolInfo)
    }
    catch
    {
        Write-Host "Failed to Convert Volume with an Exception..."
        Write-Host $_.Exception.GetType().FullName -ForegroundColor Red
        Write-Host $_.Exception.Message -ForegroundColor Red
        Write-Host "Exiting..."
        exit $1
    }
    Write-Host "Copying to destination done successfully..."
