﻿#-----------------------------------------------------------------------------------------
#
# Veritas System Recovery 16 Automation
#
# Copyright (c) 2016 by Veritas Technologies LLC
# All Rights Reserved
#
# THIS SCRIPTING EXAMPLE IS PROVIDED IN CONJUNCTION WITH YOUR LICENSE FOR VERITAS 
# SYSTEM RECOVERY SOFTWARE PRODUCTS, AND MAY BE USED ONLY IN CONJUNCTION WITH 
# THAT SOFTWARE, UNDER THE TERMS OF THE END USER LICENSE AGREEMENT THAT ACCOMPANIED 
# THE SOFTWARE. THIS SCRIPTING EXAMPLE IS PROPRIETARY VERITAS PROPERTY.  YOU MAY NOT 
# COPY, DISTRIBUTE, LICENSE OR SUBLICENSE THE SCRIPTING DOCUMENTATION.
#
#-----------------------------------------------------------------------------------------
#
#   Example Script: Consolidate all the recovery points
#
#  Objects Used: Location object, LocationConstraints, ConsolidateJob, Image objects.
#  Initialize the following variables with appropriate values:
#               1. sFilename
#               2. sIncrementalFilename
#               3. oConstraints1.ID with the Job Id
#               4. oConstraints2.ID with the Job Id
#-----------------------------------------------------------------------------------------
    
    $v2iAuto = $null
	$oNet = $null
	$oLocation1 = $null
	$oLocation2 = $null
	$oImg1 = $null
	$oImg2 = $null
	$oConstraints1 = $null
	$oConstraints2 = $null
    $aVolumes = $null
    $oTempVol = $null
    $sFilename = $null
    $sIncrementalFilename = $null
    $bVolumeFound = $null
    $oConsolidateJob = $null
    
    #----------------------------------------------------------------------------------------------------
    # Function : CreateLocation
    # Input parameters :
    #                   a)  Path can be a local folder location or a network location.
    #                   b)  Domain user name if the path is a network location.
    #                   c)  password for authenticating to the network location.
    #
    # Return parameter: Location COM object containing all the information about the location
    #-----------------------------------------------------------------------------------------------------
    
    function CreateLocation 
    {
        param ([string]$Path, [string]$domainUser, [string]$password)
        
        $oLocation = $null
        $sepPos = $null 
        $dotPos = $null
        $Path = $Path -replace "/","\"
        
        # Parse the location.
        # Check if the location is a network location or a folder location.
        # If it is a network location, set the user name and password.
        if ($Path.SubString(0,2) -eq "\\")
        {
            $oLocation = New-Object -ComObject "Veritas.VProRecovery.NetworkLocation"
            $oLocation.NetworkUser = $domainUser
            $oLocation.NetworkPassword = $password
        }
        else 
        {
            $oLocation = New-Object -ComObject "Veritas.VProRecovery.FolderLocation"
        }
        
        $sepPos = $Path.LastIndexOf("\")
        $dotPos = $Path.LastIndexOf(".")

        $oLocation.Path = [string]$Path.SubString(0, $sepPos)
        $oLocation.FileSpec = [string]$Path.SubString($sepPos+1, $dotPos - $sepPos - 1)
        $oLocation.Extension = [string]$Path.SubString($dotPos)
        
        Write-Host "Path: " $oLocation.Path 
        Write-Host "FileSpecs: " $oLocation.FileSpec 
        Write-Host "Entension: "$oLocation.Extension
        
        return $oLocation
    }
    
    #-----------------------------------------------------------------------------------------
	# Step 1: Process command line arguments.
	#-----------------------------------------------------------------------------------------
	
	if ([int]$args.Length -gt 0) 
	{
		Write-Host "Usage: ConsolidateRecoveryPoints.ps1"
		exit $1;
	}
	
	#-----------------------------------------------------------------------------------------
	# Step 2: Create a VProRecovery automation object.
	#-----------------------------------------------------------------------------------------
	
	$v2iAuto = New-Object -ComObject "Veritas.ProtectorAuto"
		
	#-----------------------------------------------------------------------------------------
	# Step 3: Connect to the local agent.
	#-----------------------------------------------------------------------------------------
	
    Write-Host "Connecting..."
	try
    {
	    $oNet = New-Object -ComObject "Wscript.Network"
        $v2iAuto.Connect($oNet.ComputerName)
    }
    catch
    {
        Write-Host "Failed to Connect with an Exception..."
        Write-Host $_.Exception.GetType().FullName -ForegroundColor Red
        Write-Host $_.Exception.Message -ForegroundColor Red
        Write-Host "Exiting..."
        exit $1
    }
    
    #-----------------------------------------------------------------------------------------------------------------------------------
    # Step 4: Define the location of the .v2i file that you want to consolidate. Specify the base filename and the incremental filename.
	#-----------------------------------------------------------------------------------------------------------------------------------
	
	$sFilename = "E:\WINVISTAB32-BAT_C_Drive.002_i001.iv2i"
	$sIncrementalFilename = "E:\WINVISTAB32-BAT_C_Drive.002_i043.iv2i"
    
    #----------------------------------------------------------------------------------------
	# Step 5: Create Location objects for the base filename and the incremental filename.
	#---------------------------------------------------------------------------------------
    
    $oLocation1 = CreateLocation $sFilename "" ""
    $oLocation2 = CreateLocation $sIncrementalFilename "" ""

    #-----------------------------------------------------------------------------------------------------------------
	# Step 6:  Define the Constraints object and set the parameters.
	#   ID - specify the Job Id/Image Id
	#   Context can be 
	#     LOCATION_WRITE	= 0,
	#     LOCATION_READ	    = 1,
	#     LOCATION_WRITE_CREATE_DIR	=2
	#   Set a password for the Image
	#   Filename is the full path of the location of the .v2i file. It can be a local folder path or a network path.
	#   RequiredSizeMB is the size in MB.
    #-------------------------------------------------------------------------------------------------------------------
    
    $oConstraints1 = New-Object -ComObject "Veritas.VProRecovery.LocationConstraints"
    $oConstraints1.ID = "{AF42CA4C-840A-4F8E-8E1B-80AAF07FC320}"
    $oConstraints1.Context = 1 #$oConstraints.Constants.LOCATION_READ
    $oConstraints1.Password = ""
    $oConstraints1.Filename = $sFilename
    $oConstraints1.RequiredSizeMB = 10245
    
    $oConstraints2 = New-Object -ComObject "Veritas.VProRecovery.LocationConstraints"
    $oConstraints2.ID = "{AF42CA4C-840A-4F8E-8E1B-80AAF07FC320}"
    $oConstraints2.Context = 1 #$oConstraints.Constants.LOCATION_READ
    $oConstraints2.Password = ""
    $oConstraints2.Filename = $sIncrementalFilename
    $oConstraints2.RequiredSizeMB = 10245
    
    #----------------------------------------------------------------------------------------------
    # Step 7: Call the ImageForFile object.
	# Input parameters a) oLocation is the Location object
	#                  b) oConstraints is the Constraints object
	# Returns the Image object
    #-----------------------------------------------------------------------------------------------
    
    #ToDo: Exception occurs while calling ImageForFile
    Write-Host "Calling ImageForFile..."
    try
    {
        $oImg1 = $v2iAuto.ImageForFile($oLocation1, $oConstraints1)
        $oImg2 = $v2iAuto.ImageForFile($oLocation2, $oConstraints2)
    }
    catch
    {
        Write-Host "Image for File failed with an Exception..."
        Write-Host $_.Exception.GetType().FullName -ForegroundColor Red
        Write-Host $_.Exception.Message -ForegroundColor Red
        Write-Host "Exiting..."
        exit $1
    }
    
    #---------------------------------------------------------------------------------------------
    # Step 8:Compare the drive letters of the Image object and the .v2i file location drive. 
	#   If true, remove the .v2i file.
    #---------------------------------------------------------------------------------------------
    
    $bVolumeFound = $false
    foreach ($oTempVol in $v2iAuto.Volumes($false))
    {
        if ($oImg1.Originaldriveletter -eq $oTempVol.GenericID)
        {
        	$bVolumeFound = $true
            break;
        }
	}
	
	if( $bVolumeFound -eq $false)
    {
	       Write-Host "Volume not found. Exiting..."
           exit $1;
    }
    
    #---------------------------------------------------------------------------------------------
    # Step 9:  Define oConsolidateJob object and set the parameters.
        #   ID - sVolumeID
        #   Reason can be 
        #       IMAGE_AUTOMATIC = 0,
        #       IMAGE_MANUAL,
        #       IMAGE_TRIGGERED
        #   Give BeginImageID
        #   Give EndImageID
    #----------------------------------------------------------------------------------------------
    
     $oConsolidateJob = New-Object -ComObject "Veritas.VProRecovery.ConsolidateJob"
     $oConsolidateJob.Volume = $oTempVol.ID
     $oConsolidateJob.BeginImage = $oImg1.ID #BeginImageID
     $oConsolidateJob.EndImage = $oImg2.ID   #EndImageID
     $oConsolidateJob.Reason = 1 #$oConsolidateJob.IMAGE_MANUAL
     
     #----------------------------------------------------------------------------------------------
      # Step 10: Call the ConsolidateIncrimentals object.
        # Input parameters 
        #           a) oConsolidateJob is the oConsolidateJob object.
     #----------------------------------------------------------------------------------------------
     
    Write-Host "Calling ConsolidateIncrimentals..."
    try
    {
        $v2iAuto.ConsolidateIncrementals($oConsolidateJob)
    }
    catch
    {
        Write-Host "Failed to Consolidate Incrementals with an Exception..."
        Write-Host $_.Exception.GetType().FullName -ForegroundColor Red
        Write-Host $_.Exception.Message -ForegroundColor Red
        Write-Host "Exiting..."
        exit $1
    }
    Write-Host "Recovery Points Consolidation Successful..."