#-----------------------------------------------------------------------------------------
#
# Veritas System Recovery 18 Automation
#
# Copyright (c) 2018 by Veritas Technologies LLC
# All Rights Reserved
#
# THIS SCRIPTING EXAMPLE IS PROVIDED IN CONJUNCTION WITH YOUR LICENSE FOR VERITAS 
# SYSTEM RECOVERY SOFTWARE PRODUCTS, AND MAY BE USED ONLY IN CONJUNCTION WITH 
# THAT SOFTWARE, UNDER THE TERMS OF THE END USER LICENSE AGREEMENT THAT ACCOMPANIED 
# THE SOFTWARE. THIS SCRIPTING EXAMPLE IS PROPRIETARY VERITAS PROPERTY.  YOU MAY NOT 
# COPY, DISTRIBUTE, LICENSE OR SUBLICENSE THE SCRIPTING DOCUMENTATION.
#
#-----------------------------------------------------------------------------------------
#   Example Script: Create an image of the system drive to a passed-in network folder.
#   Objects Used: ProtectorAuto, ImageJob, Volume, NetworkLocation , FolderLocation , oP2vSysInfo
#   Initialize the following variables with appropriate values :
#               1. oSrcNetLocation and oDestNetLocation to location       
#-----------------------------------------------------------------------------------------
  
#--------------------------------------------------------------------------------------------------------------
# PS C:> ThisPS1File
#--------------------------------------------------------------------------------------------------------------

$IsVHDRequired = $null
$Path = $null
$FolderVHD = $null
$AccessKey = $null
$SecretKey = $null
$BucketName = $null
$Region = $null
$domainUser = $null
$domainPassword = $null
$sv2iDomainUser = $null
$sv2iDomainPassword = $null
$vhdDomainUser = $null
$vhdDomainPassword = $null
$v2iAuto = $null
$oP2vSysInfo = $null
$Choice = $null
$Option = $null
$ImagePassword = $null
$folderInS3 =  $null
$flagVHD = $null
$myVersion = $null
$yes = "y"
$no = "n"

Write-Host "
        ------------------------------------------------NOTE:-----------------------------------------------
        The script converts the backup data to VHD or VHDX format so that you can create a new virtual machine 
        in Amazon cloud. 
		Ensure that you create the backup of all the drives including hidden drives for the successful creation
		of new machine.

        Please note that, if you have the installed application version lower than the Veritas System Recovery 18 
        Service Pack 1, the script supports creation of VHD or VHDX only for the backups which are 
        NOT password protected.
        However, you can create an AMI from any existing VHD or VHDX.
        ----------------------------------------------------------------------------------------------------"

#--------------------------------------------------------------------------------------------------------------
# Function to generate log
#--------------------------------------------------------------------------------------------------------------
Function LogWrite
{
Param ([string]$logstring)
if($flagVHD -eq "0")
{
    $Logfolder = "C:\ProgramData\Veritas\VERITAS SYSTEM RECOVERY\LOGS\UploadConvertVHDXLogs"
}
else
{
    $Logfolder = "C:\ProgramData\Veritas\VERITAS SYSTEM RECOVERY\LOGS\UploadConvertVHDLogs"
}
$Logfile = "$Logfolder\$env:COMPUTERNAME.log"
if((Test-Path $Logfile) -eq 0)
{
    New-Item -ItemType Directory -path $Logfolder
    New-Item -ItemType File -Path $Logfile
}
$string1 = Get-Date
Add-content $Logfile -value "$string1 : $logstring"
}

#----------------------------------------------------------------------------------------------------
# Check the current version of application
#----------------------------------------------------------------------------------------------------
$valueExists = (Get-Item "HKLM:\SOFTWARE\Veritas\Veritas System Recovery" -EA Ignore).Property -contains "Version"
if($valueExists)
{
    $myVersion = Get-ItemProperty -path "HKLM:\SOFTWARE\Veritas\Veritas System Recovery"
}
else
{
    $myVersion = Get-ItemProperty -path "HKLM:\SOFTWARE\Symantec\Symantec System Recovery"
}

#----------------------------------------------------------------------------------------------------
# Function : CreateLocation
# Input parameters :
#                   a)  Path can be a local folder location or a network location.
#                   b)  Domain user name if the path is a network location.
#                   c)  Domain password for authenticating to the network location.
#
# Return parameter: Location COM object containing all the information about the location
#-----------------------------------------------------------------------------------------------------
function CreateLocation 
{
param ([string]$Path, [string]$domainUser, [string]$domainPassword)
        
$oLocation = $null
$sepPos = $null 
$dotPos = $null
$Path = $Path -replace "/","\"
        
# Parse the location.
# Check if the location is a network location or a folder location.
# If it is a network location, set the user name and password.
if ($Path.SubString(0,2) -eq "\\")
{
    if($valueExists)
    {
        $oLocation = New-Object -ComObject "Veritas.VProRecovery.NetworkLocation"
    }
    else
    {
        $oLocation = New-Object -ComObject "Symantec.VProRecovery.NetworkLocation"
    }
    $oLocation.NetworkUser = $domainUser
    $oLocation.NetworkPassword = $domainPassword
}
else 
{
    if($valueExists)
    {
        $oLocation = New-Object -ComObject "Veritas.VProRecovery.FolderLocation"
    }
    else
    {
        $oLocation = New-Object -ComObject "Symantec.VProRecovery.FolderLocation"
    }
}

$sepPos = $Path.LastIndexOf("\")
$dotPos = $Path.LastIndexOf(".")

$oLocation.Path = [string]$Path.SubString(0, $sepPos)
$oLocation.FileSpec = [string]$Path.SubString($sepPos+1, $dotPos - $sepPos - 1)
$oLocation.Extension = [string]$Path.SubString($dotPos)
        
Write-Host "Path: " $oLocation.Path 
Write-Host "FileSpecs: " $oLocation.FileSpec 
Write-Host "Extension: "$oLocation.Extension
        
return $oLocation
}

#--------------------------------------------------------------------------------------------------------------
# Function to get bucket components
#--------------------------------------------------------------------------------------------------------------
function Get-S3folders
{
param
(
[string] $BucketName,
[string] $KeyPrefix,
[bool] $Recurse,
[string] $AccessKey,
[string] $SecretKey,
[string] $Region
)
try
{
    $Folders = $null
    [string[]]$ArrayofFolders = $null
    @(get-s3object -BucketName $BucketName -KeyPrefix $KeyPrefix -Delimiter '/' -AccessKey $AccessKey -SecretKey $SecretKey -Region $Region) | Out-Null

    if($AWSHistory.LastCommand.Responses.Last.CommonPrefixes.Count -eq 0)
    {
    return
    }

    $AWSHistory.LastCommand.Responses.Last.CommonPrefixes
    if($Recurse)
    {
    $Folders = $AWSHistory.LastCommand.Responses.Last.CommonPrefixes | % { Get-S3folders -BucketName $BucketName -KeyPrefix $_ -Recurse $Recurse -AccessKey $AccessKey -SecretKey $SecretKey -Region $Region }
    foreach($folder in $Folders)
    {
        $ArrayofFolders += $folder
    }
    }
    return $ArrayofFolders
}
catch
{
    Write-Host $_.Exception.GetType().FullName -ForegroundColor Red
    Write-Host $_.Exception.Message -ForegroundColor Red
    Write-Host "Exiting..."
    LogWrite $_.Exception.GetType().FullName
    LogWrite $_.Exception.Message
    DO
    {
        $Exit = "exit"
        $QuitApp = Read-Host -Prompt "Enter EXIT to quit the application"
    } While($QuitApp -ne $Exit)
    exit $1
}
}

#--------------------------------------------------------------------------------------------------------------
#Function to run the VHD or VHDX conversion
#--------------------------------------------------------------------------------------------------------------
function runVHDConversion
{
#-----------------------------------------------------------------------------------------
# Create a VProRecovery automation object.
#-----------------------------------------------------------------------------------------	
if($valueExists)
{
    $v2iAuto = New-Object -ComObject "Veritas.ProtectorAuto"
}
else
{
    $v2iAuto = New-Object -ComObject "Symantec.ProtectorAuto"
}

#-----------------------------------------------------------------------------------------
# Connect to the local agent.
#-----------------------------------------------------------------------------------------	
Write-Host "Connecting to the local agent."
try
{
	$oNet = New-Object -ComObject "Wscript.Network"
	$v2iAuto.Connect($oNet.ComputerName)
	Write-Host "Connected to the local agent."
}
catch
{
	Write-Host "Failed to connect with an exception. "
	Write-Host $_.Exception.GetType().FullName -ForegroundColor Red
	Write-Host $_.Exception.Message -ForegroundColor Red
	Write-Host "Exiting..."
    DO
    {
        $Exit = "exit"
        $QuitApp = Read-Host -Prompt "Enter EXIT to quit the application"
    } While($QuitApp -ne $Exit)
    exit $1
}

Write-Host "******************************************************** Virtual conversion Initiated ********************************************************"

#--------------------------------------------------------------------------------------------------------------
# Call CreateLocation to create source and destination locations
#--------------------------------------------------------------------------------------------------------------
$oSrcNetLocation = CreateLocation $Path $sv2iDomainUser $sv2iDomainPassword

$sv2iFileName = [System.IO.Path]::GetFileName($Path)
if($flagVHD -eq "0")
{
	$oDestNetLocation = CreateLocation "$FolderVHD\$sv2iFileName.vhdx" $vhdDomainUser $vhdDomainPassword
}
else
{
    $oDestNetLocation = CreateLocation "$FolderVHD\$sv2iFileName.vhd" $vhdDomainUser $vhdDomainPassword
}

#-----------------------------------------------------------------------------------------
# Call LoadSV2i to update the oP2vSysInfo object.
#-----------------------------------------------------------------------------------------

Write-Host "Loading sv2i file."
try
{
	$oP2vSysInfo =  $v2iAuto.LoadSV2i($oSrcNetLocation)
	$oP2vSysInfo.SourceLocation = $oSrcNetLocation
	$oP2vSysInfo.DestinationLocation = $oDestNetLocation
	$oP2vSysInfo.ComputerName = $sv2iFileName
    
    if($flagVHD -eq "0")
    {
		$oP2vSysInfo.DestinationConversionType = 4
    }
    else
    {
        $oP2vSysInfo.DestinationConversionType = 2
    }

	Write-Host "Successfully loaded sv2i file."
}
catch
{
	Write-Host "Failed to load the sv2i file with an exception. "
	Write-Host $_.Exception.GetType().FullName -ForegroundColor Red
	Write-Host $_.Exception.Message -ForegroundColor Red
	Write-Host "Exiting..."
    DO
    {
        $Exit = "exit"
        $QuitApp = Read-Host -Prompt "Enter EXIT to quit the application"
    } While($QuitApp -ne $Exit)
    exit $1
}

#--------------------------------------------------------------------------------------------------------------
# Call to ConvertSystem to convert to VHD or VHDX. Enter password for password protected backups
#--------------------------------------------------------------------------------------------------------------
Write-Host "Conversion in progress."
if($myVersion.Version -lt "18.0.1.0")
{
        try
        {
            $v2iAuto.ConvertSystem($oP2vSysInfo)
            Write-Host "Conversion successful."
            Write-Host "******************************************************** Virtual conversion completed ********************************************************"
        }
        catch
        {
            Write-Host "Failed to convert with an exception. "
		    Write-Host $_.Exception.GetType().FullName -ForegroundColor Red
		    Write-Host $_.Exception.Message -ForegroundColor Red
		    Write-Host "Exiting..."
            DO
            {
                $Exit = "exit"
                $QuitApp = Read-Host -Prompt "Enter EXIT to quit the application"
            } While($QuitApp -ne $Exit)
            exit $1
        }
}
else
{
    do
    {
        try
	    {
            $n = 0
            $v2iAuto.ConvertSystemFromScript($oP2vSysInfo,$ImagePassword)
            Write-Host "Conversion successful."
            Write-Host "******************************************************** Virtual conversion completed ********************************************************"
        }
        catch
	    {
            if ( $_.Exception.Message.Contains("VPRO_ERR_OPEN_IMAGE_FILE") -eq $true)
            {
                if ($ImagePassword -eq $null)
                {
                    $ImagePassword = Read-Host -AsSecureString "The image is password protected. Enter the password:"
                }
                else
                {
                    $ImagePassword = Read-Host -AsSecureString "The password is incorrect. Enter the correct password:"
                }
                $BSTR = [System.Runtime.InteropServices.Marshal]::SecureStringToBSTR($ImagePassword)            
                $ImagePassword = [System.Runtime.InteropServices.Marshal]::PtrToStringAuto($BSTR)  
                $n = 1;
            }
		    elseif ($FlagVHD -eq "0")
            {
                if( $_.Exception.GetType().FullName.Contains("InteropServices.COMException") -eq $true)
                {
                    if ($ImagePassword -eq $null)
                    {
                        $ImagePassword = Read-Host -AsSecureString "The image is password protected. Enter the password:"
                    }
                    else
                    {
                        $ImagePassword = Read-Host -AsSecureString "The password is incorrect. Enter the correct password:"
                    }
                    $BSTR = [System.Runtime.InteropServices.Marshal]::SecureStringToBSTR($ImagePassword)            
                    $ImagePassword = [System.Runtime.InteropServices.Marshal]::PtrToStringAuto($BSTR)  
                    $n = 1;
                }
            }   
            else
            {
                Write-Host "Failed to convert with an exception. "
		        Write-Host $_.Exception.GetType().FullName -ForegroundColor Red
		        Write-Host $_.Exception.Message -ForegroundColor Red
		        Write-Host "Exiting..."
                DO
                {
                    $Exit = "exit"
                    $QuitApp = Read-Host -Prompt "Enter EXIT to quit the application"
                } While($QuitApp -ne $Exit)
                exit $1
            }
        }
    }until($n -eq 0)
}
}

#--------------------------------------------------------------------------------------------------------------
# Function to create a new bucket
#--------------------------------------------------------------------------------------------------------------
function createNewBucket
{
try
{
    Write-Host "Creating the " $BucketName " bucket in " $Region " region."
    New-S3Bucket -BucketName $BucketName -Region $Region -AccessKey $AccessKey -SecretKey $SecretKey
    Write-Host "The bucket " $BucketName " in " $Region " region is successfully created."
    LogWrite "The bucket " $BucketName " in " $Region " region is successfully created."
}
catch
{
    Write-Host "Failed to create the bucket."
    Write-Host $_.Exception.GetType().FullName -ForegroundColor Red
    Write-Host $_.Exception.Message -ForegroundColor Red
    Write-Host "Exiting..."
    LogWrite "Bucket Creation Failed"
    LogWrite $_.Exception.GetType().FullName
    LogWrite $_.Exception.Message
    DO
	{
    	$Exit = "exit"
        $QuitApp = Read-Host -Prompt "Enter EXIT to quit the application"
	} While($QuitApp -ne $Exit)
    exit $1
}
}

#--------------------------------------------------------------------------------------------------------------
#Uploading .vhd files to S3 bucket
#--------------------------------------------------------------------------------------------------------------
function uploadToS3
{
if($Region -ne "us-east-1")
{
    $endpointurl = "http://s3" + $Region + ".amazonaws.com/" + $BucketName
}
if($Region -eq "us-east-1")
{
    $endpointurl = "http://s3.amazonaws.com/" + $BucketName
}

$FilePath = $FolderVHD
$FilePath = $FilePath.Replace("/","\")
if($FilePath.Substring($FilePath.Length-1) -eq "\")
{
    $FilePath = $FilePath.Substring(0, $FilePath.Length-1)
}
    
#--------------------------------------------------------------------------------------------------------------
# Get List of items in bucket
#--------------------------------------------------------------------------------------------------------------
[Array]$ListOfBucketFolders = Get-S3Object -AccessKey $AccessKey -SecretKey $SecretKey -BucketName $BucketName -Region $Region -ErrorAction SilentlyContinue

#Adding File Key from bucket to variable
$ArrayOfS3Files = $null
foreach($File in $ListOfBucketFolders)
{
    $ArrayOfS3Files += $File.Key
}

#--------------------------------------------------------------------------------------------------------------
# Get List of folders paths in destination
#--------------------------------------------------------------------------------------------------------------
[Array]$ListOfFolders1 = Get-ChildItem -Path $FilePath| ?{ $_.PSIsContainer }
    
    #Path where .vhd files are present
    $VHDPath = $FilePath
    
    #--------------------------------------------------------------------------------------------------------------
    # Getting list of files in VHD folder
    #--------------------------------------------------------------------------------------------------------------
    $files = Get-ChildItem -Path $VHDPath

    #string to check for folder in the bucket
    $compareString = $folder.Name + "/"

    #--------------------------------------------------------------------------------------------------------------
    # Upload VHD folder if not present in S3 bucket with Machine Name
    #--------------------------------------------------------------------------------------------------------------
    try
    {
        if(($ArrayOfS3Files -ccontains $compareString) -eq 0)
        {
            Write-Host "Uploading file " $VHDPath
            LogWrite "Uploading file $VHDPath"
            Write-S3Object -BucketName $BucketName -KeyPrefix $folderInS3 -Folder $VHDPath -AccessKey $AccessKey -SecretKey $SecretKey -Region $Region
            Write-Host "Successfully uploaded $VHDPath all files"
            LogWrite "Successfully uploaded $VHDPath all files"
        }
    }
    catch
    {
		Write-Host "Failed to upload the VHD files."
        Write-Host $_.Exception.GetType().FullName -ForegroundColor Red
        Write-Host $_.Exception.Message -ForegroundColor Red
        Write-Host "Exiting..."
        LogWrite $_.Exception.GetType().FullName
        LogWrite $_.Exception.Message
        LogWrite "Uploading $VHDPath to S3 bucket failed"
        DO
        {
            $Exit = "exit"
            $QuitApp = Read-Host -Prompt "Enter EXIT to quit the application"
        } While($QuitApp -ne $Exit)
        exit $1
    }
}

#--------------------------------------------------------------------------------------------------------------
# Function to create AMI in amazon
#--------------------------------------------------------------------------------------------------------------
function coversionToAMI
{
$Role = Get-IAMRole -RoleName "vmimport" -AccessKey $AccessKey -SecretKey $SecretKey -Region $Region -ErrorAction Continue
    if($Role.RoleName -eq $null)
    {
        $importPolicyDocument = @"
        {
            "Version":"2012-10-17",
            "Statement":[
                {
                    "Sid":"",
                    "Effect":"Allow",
                    "Principal":{
                    "Service":"vmie.amazonaws.com"
                    },
                    "Action":"sts:AssumeRole",
                    "Condition":{
                    "StringEquals":{
                        "sts:ExternalId":"vmimport"
                        }
                    }
                }
            ]
        }
"@
        Write-Host "Creating role vmimport as it is not available."
        LogWrite "Creating role vmimport as it is not available."
        try
        {
            New-IAMRole -RoleName vmimport -AssumeRolePolicyDocument $importPolicyDocument -AccessKey $AccessKey -SecretKey $SecretKey -Region $Region -ErrorAction Stop
            Write-Host "Role vmimport is created."
            LogWrite "Role vmimport is created." 
        }
        catch
        {
            Write-Host "Failed to create vmimport role."
        	Write-Host $_.Exception.GetType().FullName -ForegroundColor Red
        	Write-Host $_.Exception.Message -ForegroundColor Red
        	Write-Host "Exiting..."
        	LogWrite $_.Exception.GetType().FullName
        	LogWrite $_.Exception.Message
        	LogWrite "Failed to create vmimport role."
            DO
            {
                $Exit = "exit"
                $QuitApp = Read-Host -Prompt "Enter EXIT to quit the application"
            } While($QuitApp -ne $Exit)
            exit $1
        }
    }

    #--------------------------------------------------------------------------------------------------------------
    #Adding a policy to vmimport role
    #--------------------------------------------------------------------------------------------------------------
    #policy
    $rolePolicyDocument = $rolePolicyDocument = @"
{
   "Version":"2012-10-17",
   "Statement":[
      {
         "Effect":"Allow",
         "Action":[
            "s3:ListBucket",
            "s3:GetBucketLocation"
         ],
         "Resource":[
            "arn:aws:s3:::$BucketName"
         ]
      },
      {
         "Effect":"Allow",
         "Action":[
            "s3:GetObject"
         ],
         "Resource":[
            "arn:aws:s3:::$BucketName/*"
         ]
      },
      {
         "Effect":"Allow",
         "Action":[
            "ec2:ModifySnapshotAttribute",
            "ec2:CopySnapshot",
            "ec2:RegisterImage",
            "ec2:Describe*"
         ],
         "Resource":"*"
      }
   ]
}
"@
    try
    {
		Write-Host "Adding IAMRolePolicy."
        Write-IAMRolePolicy -RoleName vmimport -PolicyName vmimport -PolicyDocument $rolePolicyDocument -AccessKey $AccessKey -SecretKey $SecretKey -Region $Region
        LogWrite "Adding IAMRolePolicy."
    }
    catch
    {
        Write-Host "Failed to create IAMRolePolicy."
        Write-Host $_.Exception.GetType().FullName -ForegroundColor Red
        Write-Host $_.Exception.Message -ForegroundColor Red
        Write-Host "Exiting..."
        LogWrite $_.Exception.GetType().FullName
        LogWrite $_.Exception.Message
        LogWrite "Failed to create IAMRolePolicy."
        DO
        {
            $Exit = "exit"
            $QuitApp = Read-Host -Prompt "Enter EXIT to quit the application"
        } While($QuitApp -ne $Exit)
        exit $1
    }
    #--------------------------------------------------------------------------------------------------------------
    #Creating an Image Container Instance
    #--------------------------------------------------------------------------------------------------------------
    $windowsContainer = New-Object Amazon.EC2.Model.ImageDiskContainer
    if($flagVHD -eq "0")
    {
        $windowsContainer.Format="VHDX"
    }
    else
    {
        $windowsContainer.Format="VHD"
    }

    #--------------------------------------------------------------------------------------------------------------
    #Getting all folders in bucket
    #--------------------------------------------------------------------------------------------------------------
    write-host "Folder Name in S3 bucket is " $folderInS3
    $S3Folders =  Get-S3folders -BucketName $BucketName -KeyPrefix $folderInS3 -Recurse $true -AccessKey $AccessKey -SecretKey $SecretKey -Region $Region

    #Adding File Keys from bucket to array
    [string[]]$ArrayOfS3Folders = $null
    foreach($Folder in $S3Folders)
    {       
        $ArrayOfS3Folders += $Folder
    }
    #--------------------------------------------------------------------------------------------------------------
    #Get List of items in bucket
    #--------------------------------------------------------------------------------------------------------------
    $ListOfBucketItems = Get-S3Object -BucketName $BucketName -AccessKey $AccessKey -SecretKey $SecretKey -Region $Region

    #Adding File Keys from bucket to array
    [string[]]$ArrayOfS3Files = $null
    foreach($File in $ListOfBucketItems)
    {
        $ArrayOfS3Files += $File.Key
    }

    #Array of Image containers
    [Amazon.EC2.Model.ImageDiskContainer[]]$WindowsContainerArray = $null

    #--------------------------------------------------------------------------------------------------------------
    #S3 to EC2 conversion
    #--------------------------------------------------------------------------------------------------------------
    #Looping through the different folders in S3 bucket
    foreach($folder in $S3Folders)
    {
       if($folder -ne $null -and $folder -ne " ")
       {
        Write-Host "Creating AMI for " $folder
        LogWrite "Creating AMI for $folder"

        #Adding each image container for each .vhd to the image container array
        foreach($file in $ArrayOfS3Files)
        {
            #Checking if the file is in the concerned folder
            if($file.Contains($folder) -and $file.Length -gt $folder.Length)           
            {
                LogWrite "Adding  $file to Windows container array"
                #Details of S3 Location
                $WindowsContainer = New-Object Amazon.EC2.Model.ImageDiskContainer
                if($flagVHD -eq "0")
    			{
        			$windowsContainer.Format="VHDX"
    			}
    			else
    			{
        			$windowsContainer.Format="VHD"
    			}
                $userBucket = New-Object Amazon.EC2.Model.UserBucket
                $userBucket.S3Bucket = $bucketName
                $userBucket.S3Key = $file
                $WindowsContainer.UserBucket = $userBucket
                #Adding the Image container to array
                $WindowsContainerArray += $WindowsContainer
            }
        }
        $params = @{
                    "ClientToken"=$folder + "_" + (Get-Date)
                    "Description"=$folder + " image import " + (Get-Date)
                    "Platform"="Windows"
                    "LicenseType"="BYOL"
                    }

        try
        {
            Start-Sleep 10
            #Creating AMI adding all the .vhd files for the concerned machine
            Import-EC2Image -DiskContainer $WindowsContainerArray -AccessKey $AccessKey -SecretKey $SecretKey -Region $Region -EndpointUrl $endpointurl @params
        
            #Emptying the image container array for next iteration
            $WindowsContainerArray = $null
            write-host "AMI for $folder creation initiated successfully"
            LogWrite "AMI for $folder creation initiated successfully"
        }
        catch
        {
        	Write-Host $_.Exception.GetType().FullName -ForegroundColor Red
        	Write-Host $_.Exception.Message -ForegroundColor Red
        	Write-Host "Exiting..."
        	LogWrite $_.Exception.GetType().FullName
        	LogWrite $_.Exception.Message
        	LogWrite "Failed to create AMI for $folderInS3"
            DO
            {
                $Exit = "exit"
                $QuitApp = Read-Host -Prompt "Enter EXIT to quit the application"
            } While($QuitApp -ne $Exit)
            exit $1
        }
       }
    }
       
if($flagVHD -eq "0")
{
    $Logfolder = "C:\ProgramData\Veritas\VERITAS SYSTEM RECOVERY\LOGS\UploadConvertVHDXLogs"
}
else
{
    $Logfolder = "C:\ProgramData\Veritas\VERITAS SYSTEM RECOVERY\LOGS\UploadConvertVHDLogs"
}
$Logfile = "$Logfolder\$env:COMPUTERNAME"+"_ImportImageDetails.log"
if (Test-Path $Logfile) 
{
    Remove-Item $Logfile
}
New-Item -ItemType File -Path $Logfile
Get-EC2ImportImageTask -AccessKey $AccessKey -SecretKey $SecretKey -Region $Region | Out-File $Logfile
}

# Use existing VHDs or VHDX if present, create new ones if not.
$vhdORvhdx = Read-Host -Prompt "Do you want to proceed with using VHDX instead of VHD for AMI creation? If Yes, please Enter Y, If No, please press Enter or any key to proceed with using VHD by default"

if($vhdORvhdx -eq $yes)
{
    $flagVHD = "0" #VHDX
    Write-host "You have Entered "YES". We are proceeding with using VHDX for AMI creation"
}
else
{
    $flagVHD = "1" #VHD
    Write-host "We are proceeding with using VHD for AMI creation"
}

if($flagVHD -eq "0")
{
    Do
    {
        $IsVHDRequired = Read-host -Prompt "Do you already have a virtual hard disk - VHDX of the system drive backup (Y/N)?"
    }until($IsVHDRequired -eq $yes -or $IsVHDRequired -eq $no)
}
else
{
    Do
    {
        $IsVHDRequired = Read-host -Prompt "Do you already have a virtual hard disk - VHD of the system drive backup (Y/N)?"
    }until($IsVHDRequired -eq $yes -or $IsVHDRequired -eq $no)
}
    
if($IsVHDRequired -eq $no)
{
Write-host "You have Entered "NO" "

Do
{
    $Path  = Read-Host -Prompt "Enter the path of the sv2i file. For example: E:\Backup\machine_name_xxx.sv2i"
    $myExtension = [System.IO.Path]::GetExtension($Path)
}until($myExtension -eq ".sv2i")

if($Path.Substring($Path.Length-1) -eq "\")
{
    $Path = $Path.Substring(0, $Path.Length-1)
}
if ($Path.SubString(0,2) -eq "\\")
{
    $sv2iDomainUser = Read-Host -Prompt "Enter the network user name."

    $sv2iDomainPassword = Read-Host -Prompt "Enter the network password." -AsSecureString
    $BSTR = [System.Runtime.InteropServices.Marshal]::SecureStringToBSTR($sv2iDomainPassword)            
    $sv2iDomainPassword = [System.Runtime.InteropServices.Marshal]::PtrToStringAuto($BSTR)
}

if($flagVHD -eq "0")
{
	$FolderVHD = Read-Host -Prompt "Enter the path to create the VHDX. For example: E:\Backup\VHDX.
	You must maintain separate folders for the virtual hard disks of each machine."
}
else
{
	$FolderVHD = Read-Host -Prompt "Enter the path to create the VHD. For example: E:\Backup\VHD.
	You must maintain separate folders for the virtual hard disks of each machine."
}

if($FolderVHD.Substring($FolderVHD.Length-1) -eq "\")
{
    $FolderVHD = $FolderVHD.Substring(0, $FolderVHD.Length-1)
}

if ($FolderVHD.SubString(0,2) -eq "\\")
{
    $vhdDomainUser = Read-Host -Prompt "Enter the network user name."

    $vhdDomainPassword = Read-Host -Prompt "Enter the network password." -AsSecureString 
    $BSTR = [System.Runtime.InteropServices.Marshal]::SecureStringToBSTR($vhdDomainPassword)            
    $vhdDomainPassword = [System.Runtime.InteropServices.Marshal]::PtrToStringAuto($BSTR)
}

if($AccessKey -eq $null -and $SecretKey -eq $null)
{
    $AccessKey = Read-Host -Prompt "Enter access Key"
    $SecretKey = Read-Host -Prompt "Enter secret key" -AsSecureString
    $BSTR = [System.Runtime.InteropServices.Marshal]::SecureStringToBSTR($SecretKey)
    $SecretKey = [System.Runtime.InteropServices.Marshal]::PtrToStringAuto($BSTR)

}

Do
{
    $Choice = Read-Host 'Press 1 to create a new bucket or press 2 for an existing bucket.'
}until($Choice -eq "1" -or $Choice -eq "2")
	
if($BucketName -eq $null -and $Region -eq $null)
{
    $BucketName=Read-Host -Prompt 'Enter the bucket name.' 
    $Region = Read-Host -Prompt 'Enter the bucket region.'
}

if($folderInS3 -eq $null)
{
    $folderInS3 = Read-Host -Prompt "Enter the folder name to be created in S3 bucket to upload the files"
}

runVHDConversion
    
if($Choice -eq 1)
{
    createNewBucket 
}

uploadToS3
coversionToAMI

}

elseif($IsVHDRequired -eq $yes)
{

Write-host "You have Entered "YES" "

if($Option -eq $null)
{
    if($flagVHD -eq "0")
    {
        Do
        {
            $Option = Read-Host -Prompt 'Press 1 to upload the VHDX file to Amazon cloud before creating the AMI or Press 2 if the VHDX file is already available in S3 bucket'
            Write-Host "Your input option '$Option'"
        }until($Option -eq "1" -or $Option -eq "2")
    }
    else
    {
        Do
        {
            $Option = Read-Host -Prompt 'Press 1 to upload the VHD file to Amazon cloud before creating the AMI or Press 2 if the VHD file is already available in S3 bucket'
            Write-Host "Your input option '$Option'"
        }until($Option -eq "1" -or $Option -eq "2")
    }
}
	
if($Option -eq "1")
{
	if($FolderVHD -eq $null)
	{
        if($flagVHD -eq "0")
        {
			$FolderVHD=Read-Host -Prompt 'Enter the VHDX folder path. For example: E:\Backup\VHDX.
			You must maintain separate folders for the virtual hard disks of each machine.'
        }
        else
        {
            $FolderVHD=Read-Host -Prompt 'Enter the VHD folder path. For example: E:\Backup\VHD
			You must maintain separate folders for the virtual hard disks of each machine.'
        }
	}
    if($FolderVHD.Substring($FolderVHD.Length-1) -eq "\")
    {
        $FolderVHD = $FolderVHD.Substring(0, $FolderVHD.Length-1)
    }
}

if($AccessKey -eq $null -and $SecretKey -eq $null)
{
    $AccessKey = Read-Host -Prompt "Enter access Key"

    $SecretKey = Read-Host -Prompt "Enter secret key" -AsSecureString
    $BSTR = [System.Runtime.InteropServices.Marshal]::SecureStringToBSTR($SecretKey)
    $SecretKey = [System.Runtime.InteropServices.Marshal]::PtrToStringAuto($BSTR)
}
if($Option -eq "1")
{
    Do
    {
        $Choice = Read-Host 'Press 1 to create a new bucket or press 2 for an existing bucket.'
    }until($Choice -eq "1" -or $Choice -eq "2")
}

if($BucketName -eq $null -and $Region -eq $null)
{
	$BucketName=Read-Host -Prompt 'Enter the bucket name.'
	$Region = Read-Host -Prompt 'Enter the bucket region.'
}

if($Option -eq "1")
{
    if($folderInS3 -eq $null)
    {
        $folderInS3 = Read-Host -Prompt "Enter the folder name to upload to S3 bucket"
    }
}        
if($Choice -eq 1)
{
    createNewBucket
}

if($Option -eq "1")
{
    uploadToS3
}

if($Option -eq "2")
{
    if($flagVHD -eq "0")
    {
		$folderInS3 = Read-Host -Prompt "Enter the VHDX folder name in S3 to convert to AMI."
    }
    elseif($flagVHD -eq "1")
    {
        $folderInS3 = Read-Host -Prompt "Enter the VHD folder name in S3 to convert to AMI."
    }
}
coversionToAMI
}
DO
{
    $Exit = "exit"
    $QuitApp = Read-Host -Prompt "Enter EXIT to quit the application"
} While($QuitApp -ne $Exit)
exit $1