﻿#-----------------------------------------------------------------------------------------
#
# Veritas System Recovery 16 Automation
#
# Copyright (c) 2016 by Veritas Technologies LLC
# All Rights Reserved
#
# THIS SCRIPTING EXAMPLE IS PROVIDED IN CONJUNCTION WITH YOUR LICENSE FOR VERITAS 
# SYSTEM RECOVERY SOFTWARE PRODUCTS, AND MAY BE USED ONLY IN CONJUNCTION WITH 
# THAT SOFTWARE, UNDER THE TERMS OF THE END USER LICENSE AGREEMENT THAT ACCOMPANIED 
# THE SOFTWARE. THIS SCRIPTING EXAMPLE IS PROPRIETARY VERITAS PROPERTY.  YOU MAY NOT 
# COPY, DISTRIBUTE, LICENSE OR SUBLICENSE THE SCRIPTING DOCUMENTATION.
#
#-----------------------------------------------------------------------------------------
#
#   Example Script: Create an ImageJob to image to a passed-in local folder
#
#   Objects Used: ProtectorAuto, ImageJob, Volume, Task, FolderLocation
#   Initialize the following variables with appropriate values :
#               1. sVolume
#               2. sFolder
#-----------------------------------------------------------------------------------------
		
	$v2iAuto = $null
	$oNet = $null
	$oVolume = $null
	$oTempVol = $null
	$oTask = $null
	$oLocation = $null
	$oImageJob = $null
	$sFolder = $null

	#-----------------------------------------------------------------------------------------
	# Step 1: Process command line arguments.
	#-----------------------------------------------------------------------------------------
	
	if ([int]$args.Length -ne 1) 
	{
		Write-Host "Usage: AddImageJob-Example.ps1 [LOCAL PATH]"
		exit $1;
	}
	
    $sFolder = $args[0]
    
	#-----------------------------------------------------------------------------------------
	# Step 2: Create a VProRecovery automation object.
	#-----------------------------------------------------------------------------------------
	
	$v2iAuto = New-Object -ComObject "Veritas.ProtectorAuto"
		
	#-----------------------------------------------------------------------------------------
	# Step 3: Connect to the local agent.
	#-----------------------------------------------------------------------------------------
	
    Write-Host "Connecting..."
	 try
    {
	    $oNet = New-Object -ComObject "Wscript.Network"
        $v2iAuto.Connect($oNet.ComputerName)
    }
    catch
    {
        Write-Host "Failed to Connect with an Exception..."
        Write-Host $_.Exception.GetType().FullName -ForegroundColor Red
        Write-Host $_.Exception.Message -ForegroundColor Red
        Write-Host "Exiting..."
        exit $1
    }

	#-----------------------------------------------------------------------------------------
	# Step 4: Define the location for saving the image.  
	#-----------------------------------------------------------------------------------------
	
	$oLocation = New-Object -ComObject "Veritas.VProRecovery.FolderLocation"
	$oLocation.Path = $sFolder
	$oLocation.FileSpec = "SystemBackup"

	#-----------------------------------------------------------------------------------------
	# Step 5: Create a task to schedule the image job.
	# Parameters used:
	#                   Description - specify a description for the image job
	#                   StartDateTime - set a start date 
	#                   RepeatInterval - set an interval to repeat, such as, weekly, monthly, etc
	#                   AddRepeatDay(0) , AddRepeatDay(3) - set the days to repeat, such as Sunday, Monday, etc
	#-----------------------------------------------------------------------------------------
	
	$oTask = New-Object -ComObject "Veritas.Scheduler.Task"
	$oTask.Description = "Full Image Schedule"
	$oTask.StartDateTime = "1/1/2010 3:00"	# GMT
	$oTask.RepeatInterval = $oTask.Constants.IntervalWeekly
	$oTask.AddRepeatDay(0)	# Sunday
	$oTask.AddRepeatDay(3)	# Wednesday
	$oTask.Validate()		# Make sure we built the task correctly

	#-----------------------------------------------------------------------------------------
	# Step 6: Find the volume to image. 
	#-----------------------------------------------------------------------------------------
	
	$oVolume = $null
	foreach($oTempVol in $v2iAuto.Volumes($false))
	{
		if($oTempVol.System -eq $true)
        {
			$oVolume = $oTempVol
            break;
        }
	}
	
    If ($oVolume -eq $null)
	{
		Write-Host "AddImageJob-Example.ps1: Cannot find requested volume..."
		exit $1
	}
	
	#-----------------------------------------------------------------------------------------
	# Step 7: Create an image job.
	# Parameters used:  DisplayName - specify a display name for the image job
	#                   Description - specify a description for the image job
	#                   IncrementalSupport -  true/false 
	#                   RunOnce - true/false
	#                   Compression types
	#                        IMAGE_COMPRESSION_LOW	
	#                        IMAGE_COMPRESSION_MEDIUM	
	#                        IMAGE_COMPRESSION_HIGH	
	#                        IMAGE_COMPRESSION_NEWMEDIUM 
	#-----------------------------------------------------------------------------------------
	
	$oImageJob = New-Object -ComObject "Veritas.VProRecovery.ImageJob"
    $oImageJob.DisplayName = "Recovery point of " + $oVolume.DisplayName
	$oImageJob.Description = "Regular backup image of my system volume"
	$oImageJob.IncrementalSupport = $false
	$oImageJob.RunOnce = $false
	$oImageJob.Compression = $oImageJob.Constants.ImageCompressionLow
	$oImageJob.Reason = $oImageJob.Constants.ImageReasonAutomatic
    $oImageJob.Volumes = [Array]$oVolume.ID
    $oImageJob.Task = $oTask
	$oImageJob.Location($oVolume.ID) = $oLocation
	
    #-----------------------------------------------------------------------------------------
	# Step 8: Add the image job to the list of jobs.
	#-----------------------------------------------------------------------------------------
	
    try 
    {
	    $v2iAuto.AddImageJob($oImageJob)
    } 
    catch 
    {
        Write-Host "Add Image Job Failed with Exception..."
        Write-Host $_.Exception.GetType().FullName -ForegroundColor Red
        Write-Host $_.Exception.Message -ForegroundColor Red
        Write-Host "Exiting..."
        exit $1
    }
	Write-Host "Image Job added successfully..."