const fs = require('fs')
const _path = require('path')
const _url = require('url')

const deepProxy = (obj, handler, path = []) => {
    const proxy = new Proxy(obj, {
      get(target, prop, receiver) {
        const newPath = [...path, prop]; 
        const value = Reflect.get(target, prop, receiver);
        if (typeof value === 'object' && value !== null) {
          return deepProxy(value, handler, newPath);
        }else
        if (typeof value === 'function') {
          return function (...args) {
            return handler.apply ? handler.apply(value, args, newPath) : value.apply(target, args);
          }
        }
        return handler.get ? handler.get(target, prop, receiver, newPath) : value;
      },
      set(target, prop, value, receiver) {
        if (typeof value === 'object' && value !== null) {
          value = deepProxy(value, handler, path);
        }
        const result = Reflect.set(target, prop, value, receiver);
        return handler.set ? handler.set(target, prop, value, receiver, path) : result;
      },
      deleteProperty(target, prop) {
        const result = Reflect.deleteProperty(target, prop);
        return handler.deleteProperty ? handler.deleteProperty(target, prop, path) : result;
      },
    });
    return proxy;
 };
  
const eagle = deepProxy({...{
    folder: {},
    item: {},
    library: {
        info: () => eagle.getData('library'),
        path: electron.library_path, 
    },
}, ...electron}, {
    /*
    get(target, prop, receiver, path) {
        // switch(path.join('.')){
        //     case 'library.path':
        //         break
        // }
        return target[prop];
    },
    apply(target, args, path) {
        let callback = target.apply(this, args)
        switch(path.join('.')){
            case 'getData':
                if(args.length == 1 && args[0] == "library"){
                    callback.then(ret => {
                        if(ret.path) eagle.library.path = ret.path
                    })
                }
                break
        }
        return callback
    }
    */
})


Object.entries({folder: ['create', 'createSubfolder', 'get', 'getAll', 'getById','getSelected', 'getRecents', 'open','get', 'save'], item: ['get', 'getAll','getById', 'getSelected', 'addFromURL','addFromBase64', 'addFromPath', 'addBookmark','open', 'save']}).forEach(([type, methods]) => {
    methods.forEach(method => eagle[type][method] = (...args) => {
        return new Promise(async reslove => {
            let ret = await eagle.getData(type+'.'+method, args)
            let convert = type == 'item' ? convertToItem : convertToFolder
            if(Array.isArray(ret)){
                ret = await Promise.all(ret.map(convert))
            }else
            if(typeof(ret) == 'object'){
                ret = await convert(ret)
            }
            reslove(ret)
        })
    })
})


const isEmpty = (s, trim = false) => {
    if (s === null || typeof s === 'undefined') return true;
    if (typeof s === 'string') return (trim ? s.trim() : s).length === 0;
    return false;
}

const convertToFolder = async opts => {
    let { ctime, desc, icon, icon_color, meta, title, id } = opts
    return new Folder({
        id, name: title, icon, iconColor: icon_color, description: desc, createdAt: ctime, children: [], dirty: false, parent,
    })
}
const convertToItem = async opts => {
    let { date, id, link, md5, size, title } = opts
    let { color, desc, folders, tags, score, media, url } = await eagle.getData('getItemDetail', md5)
    let { width, height } = media || {}
    let palettes = !isEmpty(color) ? color.split('|').map(color => {
        return { color: color.split(','), ratio: 0 } // TODO 颜色比率
    }) : []
    let {ext, name} = _path.parse(title)
    return new Item({
        id: md5, name: name, ext: ext.substring(1),
        width, height, url, isDeleted: false, annotation: desc,
        tags, folders, palettes, size, star: score,
        importedAt: date, noThumbnail: false, noPreview: false
    })
}



class Folder {
    #id
    #name
    #description
    #icon
    #iconColor
    #createdAt
    #children
    #dirty

    constructor(obj) {
        if (!obj) { throw new Error("Error processing argument at index 0"); }
        this.#id = obj?.id;
        this.#name = obj?.name;
        this.#children = [];
        this.#description = obj?.description;
        this.#iconColor = obj?.iconColor;
        this.#icon = obj?.icon;
        this.#createdAt = obj?.modificationTime;
        // this.#tags = obj?.tags;
        this.#dirty = false;
        if (obj.children && obj.children.length > 0) {
            this.#children = Folder.convert(obj.children);
        }
    }

    // 保存修改
    async save() {
        return new Promise(async (resolve, reject) => {
            let result = await eagle.getData('folder.save', {
                id: this.id,
                name: this.name,
                description: this.description,
            })
            // TODO: 要有日誌
            if (result && result.id) {
                this.#dirty = false;
                return resolve(new Folder(result));
            }
            else {
                return reject(result);
            }
        });
    }

    // 🚧
    // 設定封面
    async setCover(item) {
        // TODO: 要有日誌
    }

    async open() {
        await Folder.open(this.id);
    }

    get id() { return this.#id; }

    get name() { return this.#name; }
    set name(newName) {
        if (typeof newName === 'string' && newName.length > 0) {
            this.#name = newName;
            this.#dirty = true;
        }
        else {
            throw new Error("value must be a string.");
        }
    }

    get description() { return this.#description; }
    set description(newDescription) {
        if (typeof newDescription === 'string') {
            this.#description = newDescription;
            this.#dirty = true;
        }
        else {
            throw new Error("value must be a string.");
        }
    }

    get icon() { return this.#icon; }

    get iconColor() { return this.#iconColor; }

    get createdAt() { return this.#createdAt; }

    get children() { return this.#children; }

    // get tags() { return this.#tags; }

    static async create (options) {
        return new Promise(async (resolve, reject) => {
            // TODO: 要有日誌
            let result = await eagle.getData('folder.create', options)
            if (result) {
                return resolve(new Folder(result));
            }
            return resolve(result);
        });
    }

    static async createSubfolder (parentId, options) {
        options.parent = parentId;
        return (await eagle.folder.create(options) )
    }

    static async get (options) {
        return new Promise(async (resolve, reject) => {
            let result = await eagle.getData('folder.get', options);
            return resolve(Folder.convert(result));
        });
    }

    static async getAll () { return (await eagle.folder.get()); }

    static async getById (id) { return (await eagle.folder.get({id: id}))[0];  }
    
    static async getByIds (ids) { return (await eagle.folder.get({ids: ids})); }

    static async getSelected () { return (await eagle.folder.get({ isSelected: true })); }

    static async getRecents () { return (await eagle.folder.get({ isRecent: true })); }
    
    static async open (folderId) {
        return new Promise(async (resolve, reject) => {
            let result = await eagle.getData('folder.open', { folderId });
            return resolve(result);
        });
    }

    static convert(objs) {
        try {
            let folders = [];
            objs.forEach(obj => {
                folders.push(new Folder(obj));
            });
            return folders;
        } catch (err) {
            return [];
        }
    }
}

class Item {
    #id
    #name
    #ext
    #width
    #height
    #url
    #isDeleted
    #annotation
    #tags
    #folders
    #palettes
    #size
    #star
    #importedAt
    #noThumbnail
    #noPreview
    #dirty
    constructor(obj) {
        this.#id = obj.id;
        this.#name = obj.name;
        this.#ext = obj.ext;
        this.#width = obj.width;
        this.#height = obj.height;
        this.#url = obj.url;
        this.#isDeleted = obj.isDeleted;
        this.#annotation = obj.annotation;
        this.#tags = obj.tags;
        this.#folders = obj.folders;
        this.#palettes = obj.palettes;
        this.#size = obj.size;
        this.#star = obj.star;
        this.#importedAt = obj.modificationTime;
        this.#noThumbnail = !!obj.noThumbnail;
        this.#noPreview = !!obj.noPreview;
        this.#dirty = false;
    }

    async setCustomThumbnail(filePath) {
        return new Promise(async (resolve, reject) => {
            if (!fs.existsSync(filePath)) {
                return reject(new Error(`File does not exists: ${filePath}`));
            }
            let result = await eagle.getData('item.setCustomThumbnail', {
                itemId: this.id,
                filePath: filePath
            });
            return resolve(result);
        });
    }

    async replaceFile(replaceFilePath) {
        return new Promise(async (resolve, reject) => {
            const newExt = _path.parse(replaceFilePath).ext.split(".")[1];
            let originFilePath = this.filePath;
            let changeExt = false;

            if (newExt && newExt !== this.ext) {
                this.ext = newExt;
                changeExt = true;
            }

            let tmpPath = _path.normalize(`${originFilePath}.tmp`);
            let newFilePath = this.filePath;

            try {
                eagle.log.info(`Replace item file: ${replaceFilePath} > ${originFilePath}`);

                if (!fs.existsSync(replaceFilePath)) {
                    throw Error(`File does not exists: ${replaceFilePath}`);
                }
                
                if (fs.existsSync(tmpPath)) {
                    await fs.promises.unlink(tmpPath);
                }

                await fs.promises.rename(originFilePath, tmpPath);

                await fs.promises.copyFile(replaceFilePath, newFilePath);

                if (!fs.existsSync(newFilePath) || fs.statSync(newFilePath).size === 0) {
                    throw Error(`File does not exists: ${newFilePath}`);
                }

                // update the file extension
                if (changeExt) {
                    await this.save();
                }

                // remove the backup
                try { await fs.promises.unlink(tmpPath); } catch (err) { }
            }
            catch (err) {
                // restore the file
                if (fs.existsSync(tmpPath)) {
                    await fs.promises.rename(tmpPath, originFilePath);
                }
                reject(err)
                eagle.log.error(err);
                return;
            }
            // refresh the thumbnail
            await this.refreshThumbnail();
            eagle.log.info(`Replace item file success.`);
            return resolve();
        });
    }

    async refreshThumbnail() {
        return new Promise(async (resolve, reject) => {
            // await eagle.getData('item.refreshThumbnail', {
            //     itemId: this.id
            // });
            resolve(true)
        });
    }

    async save() {
        return new Promise(async (resolve, reject) => {
            let result = await eagle.getData('item.save', {
                id: this.id,
                name: this.name,
                annotation: this.#annotation,
                url: this.#url,
                height: this.#height,
                width: this.#width,
                tags: this.#tags,
                folders: this.#folders,
                star: this.#star,
                ext: this.#ext,
            })
            console.log(result)
            // TODO: 要有日誌
            if (result && result.id) {
                return resolve(true);
            }
            else {
                return reject(false);
            }
        });
    }   

    async open() {
        await Item.open(this.id);
    }

    // 所有 setter 都要防呆、類型要一樣、不能為空值等
    get id() { return this.#id; }
    // set id(newId) { this.#id = newId; }

    get name() { return this.#name; }
    set name(newName) {
        if (typeof newName === 'string' && newName.length > 0) {
            this.#name = newName;
            this.#dirty = true;
        }
        else {
            throw new Error("value must be a string.");
        }
    }

    get ext() { return this.#ext; }
    set ext(newExt) {
        if (typeof newExt === 'string' && newExt.length > 0) {
            this.#ext = newExt;
            this.#dirty = true;
        }
        else {
            throw new Error("value must be a string.");
        }
    }

    get width() { return this.#width; }
    set width(newWidth) { 
        if (typeof newWidth === 'number') {
            this.#width = newWidth; 
            this.#dirty = true;
        }
        else {
            throw new Error("value must be numeric.");
        }
    }

    get height() { return this.#height; }
    set height(newHeight) { 
        if (typeof newHeight === 'number') {
            this.#height = newHeight;
            this.#dirty = true;
        }
        else {
            throw new Error("value must be numeric.");
        }
    }

    get url() { return this.#url; }
    set url(newURL) {
        if (
            newURL === '' ||
            typeof newURL === 'string' && newURL.match(/[-a-zA-Z0-9@:%._\+~#=]{1,256}\.[a-zA-Z0-9()]{1,6}\b([-a-zA-Z0-9()@:%_\+.~#?&//=]*)?/)
        ) {
            this.#url = newURL;
            this.#dirty = true;
        }
        else {
            throw new Error("value must be a URL or a empty string.");
        }
    }

    get isDeleted() { return this.#isDeleted; }
    // set isDeleted(bool) { 
    //     if (typeof bool === 'boolean') {
    //         this.#isDeleted = bool;
    //         this.#dirty = true;
    //     }
    //     else {
    //         throw new Error("value must be a boolean.");
    //     }
    // }

    get annotation() { return this.#annotation; }
    set annotation(newAnnotation) {
        if (typeof newAnnotation === 'string') {
            this.#annotation = newAnnotation;
            this.#dirty = true;
        }
        else {
            throw new Error("value must be a string.");
        }
    }


    get tags() { return this.#tags; }
    set tags(newTags) {
        if (Array.isArray(newTags)) {
            this.#tags = newTags;
            this.#dirty = true;
        }
    }

    get folders() { return this.#folders; }
    set folders(folderIds) {
        if (Array.isArray(folderIds)) {
            this.#folders = folderIds;
            this.#dirty = true;
        }
    }

    get palettes() { return this.#palettes; }

    get size() { return this.#size; }

    get star() { return this.#star; }
    set star(newStar) { 
        if (typeof newStar === 'number' && newStar >= 0 && newStar <= 5) {
            this.#star = newStar;
            this.#dirty = true;
        }
        else {
            throw new Error("value must be numeric & >= 0 && <= 5");
        }
    }

    get importedAt() { return this.#importedAt; }

    get noThumbnail() { return this.#noThumbnail; }
    // set noThumbnail(bool) { 
    //     if (typeof bool === 'boolean') {
    //         this.#noThumbnail = bool;
    //         this.#dirty = true;
    //     }
    //     else {
    //         throw new Error("value must be a boolean.");
    //     }
    // }

    get noPreview() { return this.#noPreview; }
    // set noPreview(bool) {
    //     if (typeof bool === 'boolean') {
    //         this.#noPreview = bool;
    //         this.#dirty = true;
    //     }
    //     else {
    //         throw new Error("value must be a boolean.");
    //     }
    // }

    get filePath() {
        return _path.normalize(`${eagle.library.path}/files/${this.id.substr(0, 2)}/${this.id.substr(2, 2)}/${this.id}/${this.name}.${this.ext}`);
    }

    get fileURL() {
        return _url.pathToFileURL(this.filePath).href;
    }

    get thumbnailPath() {
        if (this.noThumbnail) {
            return this.filePath;
        }
        else {
            return _path.normalize(`${eagle.library.path}/files/${this.id.substr(0, 2)}/${this.id.substr(2, 2)}/${this.id}/cover.jpg`);
        }
    }

    get thumbnailURL() {
        return _url.pathToFileURL(this.thumbnailPath).href;
    }

    get metadataFilePath() {
        return _path.normalize(`${eagle.library.path}/files/${this.id.substr(0, 2)}/${this.id.substr(2, 2)}/${this.id}/metadata.json`);
    }

    // ------------------------------------
    // Util methods
    // ------------------------------------

    static async get(options) {
        return new Promise(async (resolve, reject) => {
            let result = await eagle.getData('item.get', options);
            return resolve(Item.convert(result));
        });
    }

    static async getAll () {
        return (await eagle.item.get());
    }

    static async getById (id) {
        return (await eagle.item.get({ id: id }))[0];
    }

    static async getByIds (ids) {
        return (await eagle.item.get({ ids: ids }));
    }

    static async getSelected () {
        return (await eagle.item.get({ isSelected: true }));
    }

    static async #add(options) {
        return new Promise(async (resolve, reject) => {
            const { path, url, base64, name, website, annotation, tags, folders, bookmarkURL } = options;
            const itemId = eagle.guid();

            let result = await eagle.getData('item.add', {
                id: itemId,
                path: path,
                url: url,
                bookmarkURL: bookmarkURL,
                base64: base64,
                name:  name || itemId,
                website: website || "",
                annotation: annotation || "",
                tags: tags || [],  
                folders: folders || [],
            });

            if (!result) {
                return reject(false);
            }
            reslove(result.id)
        });
    }

    static async addFromBase64(base64, options = {}) {
        options.base64 = base64;
        return (await this.#add(options));
    }

    static async addFromURL(url, options = {}) {
        options.url = url;
        return (await this.#add(options));
    }

    static async addFromPath(path, options = {}) {
        options.path = path;
        return (await this.#add(options));
    }

    static async addBookmark(bookmarkURL, options = {}) {
        options.bookmarkURL = bookmarkURL;
        options.website = bookmarkURL;
        return (await this.#add(options));
    }
    
    static async open(itemId) {
        return new Promise(async (resolve, reject) => {
            let result = await eagle.getData('item.open', { itemId: itemId });
            return resolve(result);
        });
    }

    static convert(objs) {
        let items = [];
        objs.forEach(obj => {
            items.push(new Item(obj));
        });
        return items;
    }
    
}
