var fs = require("fs");

var users, groups, questions, answers, quizzes, quizSheets, answerSheets;

var studentsOfGroup = {};

var defaultGroup = {id:'_default', displayName:'默认班级'};

function newId() {
    var id = allData.id.data++;
    allData.id.dirty = true;
    return id.toString();
}

function getUser(uid) {
    return users[uid];
}

function getGroup(gid) {
    return groups[gid];
}

/**
 * 返回用户组（班级）中的学生的user对象，以map形式。
 */
function getStudentsOfGroup(gid, expand) {
	var stus = studentsOfGroup[gid];
	return stus || {};
}

function getQuiz(zid) {
	return quizzes[zid];
}

function getQuizzesOfPerson(uid) {
    var result = {};
    for ( var zid in quizzes) {
        if (quizzes[zid].teacher === uid)
            result[zid] = quizzes[zid];
    }
    return result;
}

function isQuizLive(qz) {
    if (!qz)
        return false;
    if (!qz.endTime)
        return true;
    return new Date(qz.endTime) > new Date();
}

function getLiveQuizOfPerson(id) {
    var result = {};
    //debugger;
    for ( var zid in quizzes) {
        var qz = quizzes[zid];
        if (isQuizLive(qz)) {
            if (qz.teacher === id)
                result[zid] = qz;
            else {
            	//debugger;
                var user = getUser(id);
                if (user && user.role === 'student'
                        && user.gids.indexOf(qz.group) >= 0)
                    result[zid] = qz;
            }
        }
    }
    return result;
}

function getQuestion(id) {
    return questions[id];
}

function getAllQuestions(id) {
    return questions;
}

function getAnswer(id) {
    return answers[id];
}

function getAllAnswers() {
    return answers;
}

/**
 * 更新对象的修改时间。
 */
function touch(obj) {
	//JSON.stringify(new Date()) 效果相似但是不同，
	//得到的是类似 "\"2012-09-03T08:37:32.263Z\"" 的结果，
	//需要new Date(JSON.parse(JSON.stringify(new Date()))) 才能反解。
	//toISOString() 得到 ISO8601格式，例如 "2012-09-03T08:37:32.263Z"，可以直接传入
	//Date的构造器。
    obj.mtime = new Date().toISOString(); 
}

/**
 * 修改/新建用户。
 * 
 * @param 用户的id，可选，如果不提供，将自动生成。
 * @return 用户的id。
 */
function setUser(user, id) {
	var  oldUser = users[id]; //TODO 比较新旧用户决定是否添加
	if (oldUser)
		_removeStudentFromGroup(oldUser);
	
    id = id || newId();
    touch(user);
    users[id] = user;
    allData.users.dirty = true;
    _addStudentToGroup(user);
    return id;
}

/**
 * 修改/新建用户组。
 * 
 * @param 用户组的id，可选，如果不提供，将自动生成。
 * @return 用户组的id。
 */
function setGroup(group, id) {
    id = id || newId();
    touch(group);
    groups[id] = group;
    allData.groups.dirty = true;
    return id;
}

/**
 * 修改/新建问题。
 * 
 * @param 问题的id，可选，如果不提供，将自动生成。
 * @return 问题的id。
 */
function setQuestion(ques, id) {
    id = id || newId();
    touch(ques);
    questions[id] = ques;
    allData.questions.dirty = true;
    return id;
}

/**
 * 修改/新建答案。
 * 
 * @param 答案的id，可选，如果不提供，将自动生成。
 * @return 答案的id。
 */
function setAnswer(ans, id) {
    id = id || newId();
    touch(ans);
    answers[id] = ans;
    allData.answers.dirty = true;
    return id;
}

/**
 * 修改/新建测试。
 * 
 * @param 测试的id，可选，如果不提供，将自动生成。
 * @return 测试的id。
 */
function setQuiz(quiz, id) {
    id = id || newId();
    touch(quiz);
    quizzes[id] = quiz;
    allData.quizzes.dirty = true;
    return id;
}

/**
 * 修改/新建试卷。
 * 
 * @param 问题的id，可选，如果不提供，将自动生成。
 * @return 问题的id。
 */
function setQuizSheet(quizSheet, id) {
    id = id || newId();
    touch(quizSheet);
    quizSheets[id] = quizSheet;
    allData.quizSheets.dirty = true;
    return id;
}

/**
 * 删除试卷。只做删除标记，并不真正删除。
 * TODO 级联删除试题、答卷？
 * @param zsid 试卷id
 * @returns {Boolean} 是否成功。
 */
function deleteQuizSheet(zsid) {
    if (!quizSheets[zsid])
        return false;
    else {
        quizSheets[zsid].removed = true;
        allData.quizSheets.dirty = true;
        return true;
    }    
}
exports.deleteQuizSheet = deleteQuizSheet;

/**
 * @param uid 试卷所有者的用户id。省略时返回所有试卷。
 * @returns 试卷的映射表
 */
function getQuizSheets(uid) {
//	if (!uid)
//		return quizSheets;
	var result = {};
	for (id in quizSheets) {
		var zs = quizSheets[id];
		if (uid && uid !== zs.uid)
		    continue;
		if (zs.removed)
		    continue;
		result[id] = zs;
	}
	return result;
}

/**
 * 取得试卷
 * 
 * @param id
 *            试卷id
 * @param expand
 *            可选。是否将 quizSheet.questions 由id的数组展开为question对象的数组， 是否将
 *            quizSheet.questions[i].answer 由id展开为answer对象。
 *            展开后对象的id属性将表示其id。缺省不展开。
 * @param withAnswer
 *            是否将 quizSheet.questions[i].answer 由id展开为answer对象。缺省不展开。
 * @returns quizSheet 对象。
 */
function getQuizSheet(id, expand, withAnswer) {
    var zs = quizSheets[id];
    if (!zs || !expand)
        return zs;

    var zs = clone(zs);
    zs.id = id;
    for ( var i = 0; i < zs.questions.length; i++) {
        var qid = zs.questions[i];
        var ques = clone(getQuestion(qid));
        ques.id = qid;
        zs.questions[i] = ques;
        var aid = ques.answer;
        var ans = getAnswer(aid);
        if (withAnswer && ans) {
            ques.answer = clone(ans);
            ques.answer.id = aid;
        }
    }
    return zs;
}

/**
 * 保存整个试卷。 quizSheet.questions 不是id的数组，而是question对象的数组；
 * quizSheet.questions[i].answer 不是id，而是answer对象。
 * quizSheet、question、answer对象的id属性如果存在将被用于存档，否则新生成。
 * @param uid 操作者的用户id。
 * @returns quizSheet的id。
 */
function saveQuizSheet(quizSheet, uid) {
    var questions = quizSheet.questions;
    console.log("saveQuizSheet: quizSheet =" + JSON.stringify(quizSheet));
    var qids = [];
    for ( var i = 0; i < questions.length; i++) {
        var ques = questions[i];
        if (typeof ques === 'string') {
        	 qids.push(ques);
        	continue;
        }
        var ans = ques.answer;
        delete ques.answer;
        ques.uid = uid;
        var qid = setQuestion(ques, ques.id);
        qids.push(qid);
        if (ans) {
            ans.question = qid;
            ans.uid = uid;
            var aid = setAnswer(ans, ans.id);
            ques.answer = aid;
        }
        // setQuestion(ques, qid); //也许不需要
        console.log("saveQuizSheet: ques =" + JSON.stringify(ques));
        console.log("saveQuizSheet: ans =" + JSON.stringify(ans));
    }
    quizSheet.questions = qids;
    quizSheet.uid = uid;
    console.log("saveQuizSheet: quizSheet =" + JSON.stringify(quizSheet));
    //console.log("saveQuizSheet: questions =" + JSON.stringify(questions));
    return setQuizSheet(quizSheet, quizSheet.id);
}

function getAnswerSheet(aid) {
	return answerSheets[aid];
}

function setAnswerSheet(as, id) {
    id = id || newId();
    touch(as);
    answerSheets[id] = as;
    allData.answerSheets.dirty = true;
    return id;
}

/**
 * 根据测试id和用户id得到用户的答卷
 * @param zid 测试的id
 * @param uid 用户id
 * @returns 答卷的id。要得到答卷本身，用getAnswerSheet(id)。
 * TODO: 优化性能。
 */
function getAnswerSheetViaQuizAndUser(zid, uid) {
	var ass = answerSheets;
	for (var asid in ass) {
		var as = ass[asid];
		if (as.quiz === zid && as.uid === uid)
			return asid;
	}
	return null;
}

function getAllAnswerSheetViaQuiz(zid) {
	var result = {};
	for (var asid in answerSheets) {
		var as = answerSheets[asid];
		if (as.quiz === zid)
			result[asid] = as;
	}
	return result;
}

/**
 * 选择授课班级
 * @param gid 班级id。如果为!gid为true，则取消选中的班级，data.classRoomState置为null。
 * @param uid 教师id。
 * @returns {Boolean} 是否修改了当前授课班级。
 */
function selectClass(gid, uid) {
	var user = getUser(uid);
	if (!user || user.role !== 'teacher')
		throw new Error('bad user '+uid);
	if (!gid) {
		exports.classRoomState = null;
		return true;
	}
	if (user.gids.indexOf(gid) == -1)
		throw new Error('bad gid '+gid);
	if (exports.classRoomState && exports.classRoomState.gid === gid && exports.classRoomState.gid === uid)
		return false;
	exports.classRoomState = {teacher: uid, group: gid, startTime: Date.now()};
	return true;
}

/**
 * 锁定学生端。
 * @param isLocked {Boolean} 锁定还是解锁。
 */
function lockClass(isLocked) {
	if (exports.classRoomState)
		exports.classRoomState.isLocked = !!isLocked;
}

/**
 * 新建测试
 * @param quiz 试卷对象
 * @param uid 用户的id，必须是教师
 */
function newQuiz(quiz, uid) {
	var quizSheet = getQuizSheet(quiz.quizSheet);
	var user = getUser(uid);
	if (!quizSheet)
		throw new Error('bad quiz sheet '+quizSheet);
	if (!user || user.role !== 'teacher')
		throw new Error('bad user '+uid);
	quiz.teacher = uid;
	quiz.titile = quiz.titile || quizSheet.titile;
	quiz.startTime = quiz.startTime || new Date().toISOString();
	selectClass(quiz.group, uid);
	return setQuiz(quiz);
}

/**
 * 结束所有的测试
 * @param uid 用户的id，必须是教师
 */
function endAllQuizzes(uid) {
    for (var zid in quizzes) {
        if (quizzes[zid].teacher === uid)
            endQuiz(zid);
    }
}
exports.endAllQuizzes = endAllQuizzes;

function endQuiz(zid) {
	var quiz = getQuiz(zid);
	if (!quiz)
		throw new Error('quiz not found: '+ zid);
	var now = new Date();
	if (quiz.endTime) {
		var end = new Date(quiz.endTime);
		if (now > end)
			return;
	}
	quiz.endTime = now.toISOString();
	return setQuiz(quiz, zid);
}

function userActived(uid) {
	exports.userState[uid] = Date.now();
}

function isUserActive(uid) {
	var last = exports.userState[uid];
	if (!last)
		return false;
	return Date.now() - exports.userState[uid] <= 10000;
}

function getActiveUsers() {
	var uids = [];
	var us = exports.userState;
	for (uid in us) {
		if (Date.now() - us[uid] <= 10000)
			uids.push(uid);
	}
	return uids;
} 

function _tryCreateDataDirs() {
    var dataDir = process.argv[2] || __dirname + '/data';
    var uploadDir = dataDir + '/uploads';
    exports.uploadDir = uploadDir;
    exports.dataDir = dataDir;
    if (!fs.existsSync(dataDir))
        fs.mkdirSync(dataDir);
    if (!fs.existsSync(uploadDir))
        fs.mkdirSync(uploadDir);
    for (var item in allData) {
        allData[item].fileName = dataDir + '/' + allData[item].fileName;
    }
    console.log("Using data dir: " + dataDir);
}

function _saveData() {
	//console.log(__filename + ":_saveData: started.");
	for ( var item in allData) {
		if (allData[item].dirty) {
			var fname = allData[item].fileName;
			var tmp = allData[item].fileName + '.tmp';        	
			fs.writeFileSync(tmp, JSON
					.stringify(allData[item].data), "UTF-8");
			fs.renameSync(tmp, fname);
			allData[item].dirty = false;
			console.log(__filename + ":_saveData:" + fname
					+ " saved");
		}
	}
}

function _loadData() {
    for ( var item in allData) {
    	try {
    		var content = fs.readFileSync(allData[item].fileName, "UTF-8");
    		allData[item].data = JSON.parse(content);
    		console.log(__filename + ":_loadData:" + allData[item].fileName
                    + " loaded");
    	} catch (err) {
    		 console.log(err);
    	} finally {
    		console.log(__filename + ":_loadData:" + allData[item].fileName
                    + " allData[item].data: " + allData[item].data);
    		if(allData[item].data === null || allData[item].data === undefined) {
    			if (item  === 'id')
    				allData[item].data = 10000;
    			else
    				allData[item].data = {};
    			allData[item].dirty = true;
    		} else {
    			allData[item].dirty = false;
    		}
    	}
    }
    users = allData.users.data;
    groups = allData.groups.data;
    questions = allData.questions.data;
    answers = allData.answers.data;
    quizSheets = allData.quizSheets.data;
    quizzes = allData.quizzes.data;
    answerSheets = allData.answerSheets.data;
}

function _addDefaultGroup() {
    if(!groups[defaultGroup.id])
        setGroup(defaultGroup, defaultGroup.id);
}

function _removeStudentFromGroup(user) {
	if (user.role !== "student")
		return;
	var gids = user.gids;
	for ( var i = 0; i < gids.length; i++) {
		var gid = gids[i];
		if (studentsOfGroup[gid])
			delete studentsOfGroup[gid][user.id];
	}
}

function _addStudentToGroup(user) {
	if (user.role !== "student")
		return;
	var gids = user.gids;
	for ( var i = 0; i < gids.length; i++) {
		var gid = gids[i];
		// 暂不检查 gid 是否在 groups 是否存在，允许延期登记班级信息
		if (!studentsOfGroup[gid])
			studentsOfGroup[gid] = {};
		studentsOfGroup[gid][user.id] = user;
	}
}

function _buildStudentsOfGroup() {
    for ( var uid in users) {
    	var user = users[uid];
        user.id = uid; //TODO：统一改
        _addStudentToGroup(user);
    }
}

function clone(obj) {
    if (obj === undefined || obj === null || !(obj instanceof Object))
        return obj;
    try {
        return JSON.parse(JSON.stringify(obj));
    } catch (e) {
        console.log("clone: error: obj = " + JSON.stringify(obj));
        throw e;
    }
}

var allData = {
    questions : {
        fileName : "questions.json",
        data : null,
        dirty : false
    },
    answers : {
        fileName : "answers.json"
    },
    users : {
        fileName : "users.json"
    },
    groups : {
        fileName : "groups.json"
    },
    quizSheets : {
        fileName : "quiz-sheets.json"
    },
    quizzes : {
        fileName : "quizzes.json"
    },
    answerSheets : {
        fileName : "answer-sheets.json"
    },
    id : {
        fileName : "id.json"
    }
};


_tryCreateDataDirs();
_loadData();
_addDefaultGroup();
_buildStudentsOfGroup();

process.on('exit', function() {
    console.log(__filename + "process.on('exit'):1");
});

process.on('exit', function() {
    console.log(__filename + "process.on('exit'):2");
    _saveData();
});

setInterval(_saveData, 5400);

exports.getUser = getUser;
exports.setUser = setUser;
exports.getGroup = getGroup;
exports.setGroup = setGroup;
exports.getStudentsOfGroup = getStudentsOfGroup;
exports.getQuestion = getQuestion;
exports.getAnswer = getAnswer;
exports.getQuizSheet = getQuizSheet;
exports.getAllQuestions = getAllQuestions;
exports.saveQuizSheet = saveQuizSheet;
exports.getAnswerSheet = getAnswerSheet;
exports.setAnswerSheet = setAnswerSheet;
exports.getAnswerSheetViaQuizAndUser = getAnswerSheetViaQuizAndUser;
exports.getAllAnswerSheetViaQuiz = getAllAnswerSheetViaQuiz;
exports.getQuizSheets = getQuizSheets;
exports.getQuiz = getQuiz;
exports.newQuiz = newQuiz;
exports.endQuiz = endQuiz;
exports.getLiveQuizOfPerson = getLiveQuizOfPerson;
exports.sync = _saveData;
exports.clone = clone;
exports.newId = newId;
exports.defaultGroup = defaultGroup;
exports.classRoomState = null; //{teacher:string/uid, group:string/gid, startTime:string/ISODate}
exports.userState = {}; //{<uid(string), lastActiveTime(number)>}
exports.userActived = userActived;
//exports.isUserActive = isUserActive;
exports.getActiveUsers = getActiveUsers;
exports.selectClass = selectClass;
exports.lockClass = lockClass;
