var express = require('express');
var fs = require("fs");
// var cookies = require('cookies');

var data = require('./data');

var app = express();

// 定义共享环境
app.configure(function() {
    app.use(express.methodOverride());
    app.use(express.cookieParser()); // cookieParser 要在 bodyParser
                                        // 之前，否则cookie不会被解析，req.cookies是undefined。
    app.use(express.session({
        secret : "keyboard cat"
    }));
    app.use(express.bodyParser());
    app.use(app.router);
});

// 定义开发环境
app.configure('development', function() {
    app.use(express.static(__dirname + '/public'));
    app.use(express.static(data.uploadDir));
    app.use(express.errorHandler({
        dumpExceptions : true,
        showStack : true
    }));
});

// 定义生产环境
app.configure('production', function() {
    var oneYear = 31557600000;
    app.use(express.static(__dirname + '/public', {
        maxAge : oneYear
    }));
    app.use(express.static(data.uploadDir, {
        maxAge : oneYear
    }));
    app.use(express.errorHandler());
});

app.get('/', function(req, res) {
    res.send('hello world from csser.com!');
});

/**
 * 参数是id、role、displayName、sex（'m'|'f'）、
 * gids（用户组的数组，逗号分隔）。
 * 如果用户id不存在而role存在，将新建用户并登录；如果用户id存在，则更新用户信息。
 */
app.get('/login/:id', function(req, res) {
    var uid = req.params.id;
    var user = data.getUser(uid);
    var newUser = false;
    var update = false;
    
    if (!user && (req.query.role === 'student' || req.query.role === 'teacher')) {
    	newUser = true;
    	user = {id:uid};
    }
    
    if (!user) {
    	res.cookie("uid", "");
    	res.status(403);
        res.send('not found id ' + uid);
        console.log("app.get('/login/:id'):  not found id:"+ uid);
        return;
    }
    
    if (req.query.role === 'student' || req.query.role === 'teacher') {
    	user.role = req.query.role;
    	update = true;
    }
    
    var myGids = null;
    if (req.query.gids) {
    	myGids = req.query.gids.split(',');
    	user.gids = myGids;
    	update = true;
    }
    user.gids = user.gids || [data.defaultGroup.id];
    
    if (req.query.sex) {
    	user.sex = req.query.sex;
    	update = true;
    }

    if (req.query.displayName) {
    	user.displayName =  req.query.displayName;
    	update = true;
    }

    if (update || newUser) {
    	console.log("app.get('/login/:id'):  created/updated user:"+ JSON.stringify(user));
    	data.setUser(user, uid);
    }

    res.cookie("uid", uid);
    if (newUser)
    	res.send('user created');
    else
    	res.send('login ok');
});

app.get('/shutdown', function(req, res) {
    var uid = req.cookies.uid;
    var user;
    if (uid && (user = data.getUser(uid)) && user.role == 'teacher') {
        data.sync();
        res.send('shutdown ok ');
    } else {
        res.status(403);
        res.send('no such user or no permission: ' + uid);
    }
});

app.get('/groups', function(req, res) {
	var uid = req.cookies.uid;
	var user;
	if (!uid || !(user = data.getUser(uid)) ) {
		res.status(403);
        res.send('no such user or no permission: ' + uid);
        return;
	}
	var groups = {};
	for (var i=0; i<user.gids.length; i++) {
		groups[user.gids[i]] = data.getGroup(user.gids[i]);		
	}
	res.send(groups);
});

app.post('/updateGroups', function(req, res) {
	var uid = req.cookies.uid;
	if (!isTeacher(req) ) {
		res.status(403);
        res.send('no such user or no permission: ' + uid);
        return;
	}
	var grpList = req.body;
	console.log("/updateGroups: body: "+JSON.stringify(req.body));
	if (!(grpList instanceof Array)) {
		res.status(401);
        res.send('bad data format; array of group expected. ' );
        console.log('bad data format; array of group expected. ');
        return;
	}
	for (var i=0; i < grpList.length; i++) {
		data.setGroup(grpList[i], grpList[i].id);
	}
	res.send(200);
});

app.get('/studentsOfGroup/:gid', function(req, res) {
	var uid = req.cookies.uid;
	if (!isTeacher(req) ) {
		res.status(403);
        res.send('no such user or no permission: ' + uid);
        return;
	}
	var gid = req.params.gid;
	var grp = data.getGroup(gid);
	if (!grp) {
		res.send(404);
		return;
	}
	var students = data.getStudentsOfGroup(gid);	
	res.send(students);
});

app.get('/activeUsers', function(req, res) {
	if (!isTeacher(req) ) {
		res.status(403);
        res.send('no such user or no permission: ' + uid);
        return;
	}	
	var uids = data.getActiveUsers();
	res.send(uids);
});

app.get('/questions', function(req, res) {
    var uid = req.cookies.uid;
    var user;
    if (uid && (user = data.getUser(uid)) && user.role == 'teacher') {
        var ret = {};
        ret.questions = data.getAllQuestions();
        ret.answers = {};
        for ( var qid in ret.questions) {
            var aid = ret.questions[qid].answer;
            if (aid)
                ret.answers[aid] = data.getAnswer(aid);
        }
        res.contentType("json"); // or application/json
        res.send(JSON.stringify(ret)); // or just res.send(ret)
    } else {
        res.status(403);
        res.send('no such user or no permission: ' + uid);
    }
});

app.get('/quizSheets', function(req, res) {
    var zsid = req.params.zsid;
    if (!isTeacher(req)) {
        res.status(403);
        res.send('no such user or no permission.');
        return;
    }
    res.send(data.getQuizSheets(/*req.cookies.uid*/)); //反注释 req.cookies.uid 可启用根据用户id过滤试卷
});

app.get('/quizSheet/:zsid', function(req, res) {
    var zsid = req.params.zsid;
    if (!isUser(req)) {
        res.status(403);
        res.send('no such user ');
        return;
    }
    var expand = true;
    //var withAnswer = isTeacher(req); // TODO 根据测试的进行决定是否发送答案
    var withAnswer = true; // TODO 根据测试的进行决定是否发送答案
    var result = {};
    var result = data.getQuizSheet(zsid, expand, withAnswer);
    if (!result) {
        res.status(404);
        res.send('no such quiz sheet id: ' + zsid);
        return;
    }
    res.send(result);
});

///TODO 用delete方法？
app.get('/deleteQuizSheet/:zsid', function(req, res) {
    var zsid = req.params.zsid;
    if (!isUser(req)) {
        res.status(403);
        res.send('no such user ');
        return;
    }
    var result = data.deleteQuizSheet(zsid);
    if (!result) {
        res.status(404);
        res.send('no such quiz sheet id: ' + zsid);        
    } else {
        res.send('delete ok');
    }
});

app.post('/saveQuizSheet', function(req, res) {
    if (!isTeacher(req)) {
        res.status(403);
        res.send('no such user or no permission');
        return;
    }
    var quizSheet = req.body;
    var zid = data.saveQuizSheet(quizSheet, req.cookies.uid);
    res.status(200);
    res.send(zid);
    data.sync();
});

// / 处理图像上传，与nicEdit兼容。
app.post('/uploadImage', function(req, res) {
    if (!isUser(req)) {
        res.send(403);
        return;
    }
    if (req.files && req.files.image) {
        var image = req.files.image;
        // windows 上临时目录与"/data/uploads"可能不在同一驱动器，所以rename可能失败，所以用复制的办法。
        fs.readFile(image.path, function(err, fdata) {
            if (err) {
                res.send(500);
                return;
            }
            var imageUrl = "/" + data.newId() + image.filename;
            var newPath = data.uploadDir + imageUrl;
            console.log("/uploadImage: new path:" + newPath);
            fs.writeFile(newPath, fdata, function(err) {
                if (err) {
                    res.send(500);
                    return;
                }
                var ret = {
                    upload : {
                        links : {
                            original : imageUrl
                        }
                    }
                };
                res.send(ret);
            });
        });
        console.log("/uploadImage: " + JSON.stringify(req.files));
    } else {
        res.send(406); // Not Acceptable
    }
});

app.post('/selectClass', function(req, res) {
	var uid = req.cookies.uid;
	if (!isTeacher(req)) {
        res.status(403);
        res.send('no such user or no permission');
        return;
    }
	var gid = req.body;
	console.log('/selectClass: gid=' + JSON.stringify(gid));
	data.selectClass(gid.gid, uid);
	res.send('ok');
});

app.post('/lockClass', function(req, res) {
	var uid = req.cookies.uid;
	if (!isTeacher(req)) {
        res.status(403);
        res.send('no such user or no permission');
        return;
    }
	var locked = req.body.locked;
	data.lockClass(locked);
	res.send('ok');
});

app.get('/quiz/:zid', function(req, res) {
	var zid = req.params.zid;
	var result = data.getQuiz(zid);
	if (!result) {
		res.send(404);
		return;
	}
	result = data.clone(result);
	result.students = data.getStudentsOfGroup(result.group);
    res.send(result); 
    console.log("app.get('/quiz/:zid'):" + zid);
});

app.post('/newQuiz', function(req, res){
	var uid = req.cookies.uid;
	if (!isTeacher(req)) {
        res.status(403);
        res.send('no such user or no permission');
        return;
    }
	var newQuiz = req.body;
	var zid = data.newQuiz(newQuiz, uid);
	res.send(zid);
	data.sync();
});

app.get('/endQuiz/:zid', function(req, res){
	var uid = req.cookies.uid;
	if (!isTeacher(req)) {
        res.status(403);
        res.send('no such user or no permission');
        return;
    }
	var zid = req.params.zid;
	//data.endQuiz(zid); //这个方法可能导致残留的测试存在。TODO：提供历史测试列表。
	data.endAllQuizzes(uid);
	res.send(200);
	data.sync();
});

app.get('/liveQuizzes', function(req, res) {
	if (!isUser(req)) {
        res.status(403);
        res.send('no such user or no permission');
        return;
    }
    var uid = req.cookies.uid;
	var quizzes = data.getLiveQuizOfPerson(uid);
	res.setHeader('Cache-Control', 'no-cache');
    res.send(quizzes); 
    data.userActived(uid);
});

app.get('/user/:hisUid', function(req, res) {
    var uid = req.cookies.uid;
    var hisUid = req.params.hisUid; //TODO: 根据id关系限制访问
	var user = data.getUser(hisUid);
	if (user)
		res.send(user);
	else
		res.send(404);
});

app.get('/answerSheet/viaQuiz/:zid', function(req, res) {
	var zid = req.params.zid;
    var uid = req.cookies.uid;
	var as = data.getAnswerSheet(data.getAnswerSheetViaQuizAndUser(zid, uid));
    res.contentType("json"); 
    res.send(JSON.stringify(as)); 
});

app.get('/answerSheet/allViaQuiz/:zid', function(req, res) {
	var zid = req.params.zid;
    if (!isTeacher(req)) {
        res.status(403);
        res.send('no such user or no permission');
        return;
    }
    res.send(data.getAllAnswerSheetViaQuiz(zid)); 
});

app.post('/postAnswerSheet', function(req, res) {
    var answerSheet = req.body;
    console.log('/postAnswerSheet:' + JSON.stringify(answerSheet));
    var uid = req.cookies.uid;
    var oldAsId = data.getAnswerSheetViaQuizAndUser(answerSheet.quiz, uid);
    var oldAs = data.getAnswerSheet(oldAsId);
    if (oldAs && oldAs.commit) {
        res.status(200);
        res.send('ignored');
        return;
    }
    answerSheet.uid = uid;
    data.setAnswerSheet(answerSheet, oldAsId);
    res.status(200);
    res.send('ok');
});

// utility funtions
function isUser(req) {
    var uid = req.cookies.uid;
    var user;
    return !!(uid && (user = data.getUser(uid)));
}

function isTeacher(req) {
    var uid = req.cookies.uid;
    var user;
    return !!(uid && (user = data.getUser(uid)) && user.role == 'teacher');
}

function isStudent(req) {
    var uid = req.cookies.uid;
    var user;
    return !!(uid && (user = data.getUser(uid)) && user.role == 'student');
}

app.listen(4088);

// udp

var dgram = require('dgram');
var os = require('os');
var sourcePort = 43075;
var destPort = 43001;

function broadcastClassRoom() {

	//udp socket.close() can throw, so we have to trap it.
	function close(socket) {
		try {
			socket.close();
		} catch (err) {
			console.log("broadcastClassRoom: close: " + err.stack);
		}
	}

	setTimeout(broadcastClassRoom, 3000);

	if (!data.classRoomState) {
		console.log("broadcastClassRoom: no class selected");
		return;
	}
	var strMsg = JSON.stringify(data.classRoomState);
	////var strMsg = "aa"
	var message = new Buffer(strMsg);
	var nics = os.networkInterfaces();
	for(nicName in nics) {
		var addrs = nics[nicName];
		addrs.forEach(function(item){
			if (item.family === 'IPv4' && !item.internal) {
				var client = dgram.createSocket("udp4");
				// 'error' event is the only way to trap bind errors except global uncaughtException handler
				client.on('error', function(err) {
					close(client);
					console.log("broadcastClassRoom: on error: addr = " + item.address + ", err = " + err.stack);
				});	
				client.on('listening', function() {
					console.log("broadcastClassRoom: send: "+  strMsg);					
					try {
						client.setBroadcast(true); //must not be called before binding complete.
						// dgram.Socket.send can also throw... crazy.
						client.send(message, 0, message.length, destPort, "255.255.255.255", function(err, bytes) {
							if (err)
								console.log("broadcastClassRoom: send error: addr = " + item.address + ", err = " + err.stack);
							close(client);
						});	
					} catch(ex) {
						console.log("broadcastClassRoom: socket error: " + ex.stack);
						close(client);
						return;
					}
				});	
				//console.log("broadcastClassRoom: binding "+  item.address);
				client.bind(sourcePort, item.address); //if bind ok, 'listening' event will be raised.					
			}
		});
	}
}

function receiveClassRoomActivity() {
	var server = dgram.createSocket("udp4");
	server.on("message", function (msg, rinfo) {
		var uid = msg.toString();
		console.log("receiveClassRoomActivity got: " + uid + " from " + rinfo.address + ":" + rinfo.port);
		data.userActived(uid);
	});

	server.on("listening", function () {
		var address = server.address();
		console.log("receiveClassRoomActivity: listening " +
				address.address + ":" + address.port);
	});

	server.on('error', function(err) {
		close(server);
		console.log("receiveClassRoomActivity: on error:  " + err.stack);
		//setTimeout(receiveClassRoomActivity, 10);
	});	
	
	server.bind(sourcePort + 1);
}

broadcastClassRoom();
receiveClassRoomActivity();
